// SPDX-License-Identifier: GPL-2.0-or-later
// Copyright (c) 2012-2020 Association Prologin <association@prologin.org>

// This file contains all the API functions for the C language, and all the
// constants, enumerations and structures.
// This file was generated by stechec2-generator. DO NOT EDIT.

#pragma once

#include <stdbool.h>
#include <stdlib.h>

/// Nombre de lignes dans la carte
#define HAUTEUR 40

/// Nombre de colonnes dans la carte
#define LARGEUR 40

/// Nombre de tours à jouer avant la fin de la partie
#define NB_TOURS 400

/// Taille de départ d'une troupe
#define TAILLE_DEPART 5

/// Taille minimale qu'une troupe peut avoir avant de se disperser
#define TAILLE_MIN 3

/// Nombre de troupes que chaque joueur controle
#define NB_TROUPES 2

/// Intervalle de distribution de pains par les papys
#define INTERVALLE_DISTRIB 5

/// Nombre de tunnels qu'un joueur peut creuser par tour
#define FREQ_TUNNEL 1

/// Nombre de déplacements que peut faire une troupe en un tour
#define PTS_ACTION 5

/// Nombre de points de mouvement requis pour incrémenter la taille
#define COUT_CROISSANCE 3

/// Coût en score de la pose de buisson
#define COUT_BUISSON 3

/// Round à la fin duquel les barrières s'ouvrent ou se ferment
#define ROUND_FERMETURE 99

/// Erreurs possibles après avoir effectué une action
typedef enum erreur
{
    OK, ///< L'action a été effectuée avec succès
    TROUPE_INVALIDE, ///< Mauvais identifiant de troupe
    HORS_TOUR, ///< Aucune action n'est possible hors de joueur_tour
    MOUVEMENTS_INSUFFISANTS, ///< Il ne reste plus assez de points de mouvements pour effectuer l'action demandée
    TROP_GRANDI, ///< La troupe a déjà trop grandi pendant le tour
    TROP_CREUSE, ///< Trop de trous ont déjà été creusés pendant le tour
    NON_CREUSABLE, ///< Il n'est pas possible de creuser à la position demandée
    NON_CONSTRUCTIBLE, ///< La zone demandée n'est pas constructible
    SCORE_INSUFFISANT, ///< Le joueur n'a pas assez de points pour construire un buisson
    POSITION_INVALIDE, ///< La position demandée est hors du parc
    DIRECTION_INVALIDE, ///< La direction spécifiée n'existe pas.
    PIGEON_INVALIDE, ///< Le pigeon spécifié n'existe pas.
} erreur;

/// Directions possibles
typedef enum direction
{
    NORD, ///< Sens positif pour les lignes
    SUD, ///< Sens négatif pour les lignes
    EST, ///< Sens positif pour les colonnes
    OUEST, ///< Sens négatif pour les colonnes
    HAUT, ///< Sens positif pour le niveau
    BAS, ///< Sens négatif pour le niveau
} direction;

/// Type de l'élément présent sur une case
typedef enum type_case
{
    GAZON, ///< Absence d'élément
    BUISSON, ///< Obstacle impossible à traverser
    BARRIERE, ///< Élément pouvant être ouvert ou fermé. Une barrière fermée est infranchissable alors qu'une barrière ouverte est analogue à une case vide
    NID, ///< Élément traversable permettant à la troupe de déposer son inventaire en échange de points
    PAPY, ///< Élément traversable générant de manière périodique des miches de pain
    TROU, ///< Interface entre le niveau principal est le niveau souterrain
    TUNNEL, ///< Bloc du souterrain ayant été creusé
    TERRE, ///< Bloc du souterrain n'ayant pas encore été creusé
} type_case;

/// État d'une barrière, soit ouvert, soit fermé, soit non-applicable
typedef enum etat_barriere
{
    OUVERTE, ///< La barrière est ouverte
    FERMEE, ///< La barrière est fermée
    PAS_DE_BARRIERE, ///< L'élément dont on requiert l'état n'est pas une barrière
} etat_barriere;

/// Joueur auquel appartient un nid
typedef enum etat_nid
{
    LIBRE, ///< Le nid n'a pas été attribué
    JOUEUR_0, ///< Joueur 0
    JOUEUR_1, ///< Joueur 1
    PAS_DE_NID, ///< L'élément dont on requiert l'état n'est pas un nid
} etat_nid;

/// Type de pigeon de debug
typedef enum pigeon_debug
{
    PAS_DE_PIGEON, ///< Aucun pigeon, enlève le pigeon présent
    PIGEON_BLEU, ///< Pigeon bleu
    PIGEON_JAUNE, ///< Pigeon jaune
    PIGEON_ROUGE, ///< Pigeon rouge
} pigeon_debug;

/// Types d'actions
typedef enum type_action
{
    ACTION_AVANCER, ///< Action ``avancer``
    ACTION_GRANDIR, ///< Action ``grandir``
    ACTION_CONSTRUIRE, ///< Action ``construire buisson``
    ACTION_CREUSER, ///< Action ``creuser tunnel``
} type_action;

struct position;
struct troupe;
struct etat_case;
struct action_hist;

typedef struct action_hist_array
{
    struct action_hist* items;
    size_t length;
} action_hist_array;

typedef struct direction_array
{
    direction* items;
    size_t length;
} direction_array;

typedef struct position_array
{
    struct position* items;
    size_t length;
} position_array;

typedef struct troupe_array
{
    struct troupe* items;
    size_t length;
} troupe_array;

/// Position dans la carte, donnée par trois coordonnées
typedef struct position
{
    int colonne; ///< Abscisse
    int ligne; ///< Ordonnée
    int niveau; ///< Niveau
} position;

/// Une troupe, composée de la maman canard et de ses canetons
typedef struct troupe
{
    position maman; ///< Position de la maman canard
    position_array canards; ///< Position des différents canards de la troupe, incluant la maman en première position
    int taille; ///< Taille de la troupe
    direction dir; ///< Direction de la troupe
    int inventaire; ///< Nombre de pains de la troupe
    int pts_action; ///< Nombre de points d'action de la troupe
    int id; ///< Identifiant de la troupe
} troupe;

/// Élément constituant le parc
typedef struct etat_case
{
    position pos; ///< Position de la case. Le niveau vaut nécessairement 0
    type_case contenu; ///< Type de la case
    bool est_constructible; ///< La case est constructible
    int nb_pains; ///< Nombre de pains contenus sur la case
} etat_case;

/// Action représentée dans l'historique
typedef struct action_hist
{
    type_action action_type; ///< Type de l'action
    int troupe_id; ///< Identifiant de la troupe
    direction action_dir; ///< Direction de l'action
    position action_pos; ///< Position de l'action
} action_hist;

// La troupe avance d'une case vers une direction donnée
erreur avancer(int id, direction dir);

// La troupe grandit
erreur grandir(int id);

// Construit un buisson à la position donnée
erreur construire_buisson(position pos);

// Creuse un tunnel à la position donnée
erreur creuser_tunnel(position pos);

// Renvoie les informations concernant une case
etat_case info_case(position pos);

// Renvoie les informations d'état d'une barrière
etat_barriere info_barriere(position pos);

// Renvoie les informations d'état d'un nid
etat_nid info_nid(position pos);

// Renvoie le nombre de tours restants avant qu'un papy dépose une miche de
// pain. Retourne -1 si aucun papy ne se trouve à la position demandée
int papy_tours_restants(position pos);

// Renvoie les troupes d'un joueur. Si le joueur est invalide, tous les champs
// valent -1.
troupe_array troupes_joueur(int id_joueur);

// Renvoie la position des pains récupérables
position_array pains(void);

// Pose un pigeon de debug sur la case indiquée
erreur debug_poser_pigeon(position pos, pigeon_debug pigeon);

// Renvoie la liste des actions effectuées par l'adversaire durant son tour,
// dans l'ordre chronologique. Les actions de débug n'apparaissent pas dans
// cette liste.
action_hist_array historique(void);

// Renvoie le gain en score que le nombre de pains passé en entrée rapporterait
// s'ils étaient tous déposés d'un coup dans un nid
int gain(int nb_pains);

// Renvoie la taille de l'inventaire d'une troupe de taille donnée
int inventaire(int taille);

// Trouve un plus court chemin ouvert entre deux positions. Renvoie une liste
// vide si les deux positions sont égales ou si aucun chemin n'existe.
direction_array trouver_chemin(position depart, position arrivee);

// Renvoie votre numéro de joueur.
int moi(void);

// Renvoie le numéro du joueur adverse.
int adversaire(void);

// Renvoie le score du joueur `id_joueur`. Renvoie -1 si le joueur est
// invalide.
int score(int id_joueur);

// Annule la dernière action. Renvoie faux quand il n'y a pas d'action à
// annuler ce tour-ci
bool annuler(void);

// Retourne le numéro du tour actuel.
int tour_actuel(void);

// Affiche le contenu d'une valeur de type erreur
void afficher_erreur(erreur v);

// Affiche le contenu d'une valeur de type direction
void afficher_direction(direction v);

// Affiche le contenu d'une valeur de type type_case
void afficher_type_case(type_case v);

// Affiche le contenu d'une valeur de type etat_barriere
void afficher_etat_barriere(etat_barriere v);

// Affiche le contenu d'une valeur de type etat_nid
void afficher_etat_nid(etat_nid v);

// Affiche le contenu d'une valeur de type pigeon_debug
void afficher_pigeon_debug(pigeon_debug v);

// Affiche le contenu d'une valeur de type type_action
void afficher_type_action(type_action v);

// Affiche le contenu d'une valeur de type position
void afficher_position(position v);

// Affiche le contenu d'une valeur de type troupe
void afficher_troupe(troupe v);

// Affiche le contenu d'une valeur de type etat_case
void afficher_etat_case(etat_case v);

// Affiche le contenu d'une valeur de type action_hist
void afficher_action_hist(action_hist v);
