// SPDX-License-Identifier: GPL-2.0-or-later
// Copyright (c) 2012-2020 Association Prologin <association@prologin.org>
//
// This file contains all the API functions for the Rust language, and all the
// constants, enumerations and structures.
//
// It has been generated. You can modify the end of the file if you want, but
// do not touch the part where constants, enums, structs, and api functions are
// defined.

//! Rust API for prologin2022

use crate::ffi;
use crate::ffi::{array_of_borrow_to_c, CToRust, RustToC};

#[allow(unused_imports)]
use std::{cell::UnsafeCell, borrow::Borrow};

/// Nombre de lignes dans la carte
pub const HAUTEUR: i32 = 40;

/// Nombre de colonnes dans la carte
pub const LARGEUR: i32 = 40;

/// Nombre de tours à jouer avant la fin de la partie
pub const NB_TOURS: i32 = 400;

/// Taille de départ d'une troupe
pub const TAILLE_DEPART: i32 = 5;

/// Taille minimale qu'une troupe peut avoir avant de se disperser
pub const TAILLE_MIN: i32 = 3;

/// Nombre de troupes que chaque joueur controle
pub const NB_TROUPES: i32 = 2;

/// Intervalle de distribution de pains par les papys
pub const INTERVALLE_DISTRIB: i32 = 5;

/// Nombre de tunnels qu'un joueur peut creuser par tour
pub const FREQ_TUNNEL: i32 = 1;

/// Nombre de déplacements que peut faire une troupe en un tour
pub const PTS_ACTION: i32 = 5;

/// Nombre de points de mouvement requis pour incrémenter la taille
pub const COUT_CROISSANCE: i32 = 3;

/// Coût en score de la pose de buisson
pub const COUT_BUISSON: i32 = 3;

/// Round à la fin duquel les barrières s'ouvrent ou se ferment
pub const ROUND_FERMETURE: i32 = 99;

/// Erreurs possibles après avoir effectué une action
#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq, PartialOrd, Ord)]
pub enum Erreur {
    /// L'action a été effectuée avec succès
    Ok,
    /// Mauvais identifiant de troupe
    TroupeInvalide,
    /// Aucune action n'est possible hors de joueur_tour
    HorsTour,
    /// Il ne reste plus assez de points de mouvements pour effectuer l'action
    /// demandée
    MouvementsInsuffisants,
    /// La troupe a déjà trop grandi pendant le tour
    TropGrandi,
    /// Trop de trous ont déjà été creusés pendant le tour
    TropCreuse,
    /// Il n'est pas possible de creuser à la position demandée
    NonCreusable,
    /// La zone demandée n'est pas constructible
    NonConstructible,
    /// Le joueur n'a pas assez de points pour construire un buisson
    ScoreInsuffisant,
    /// La position demandée est hors du parc
    PositionInvalide,
    /// La direction spécifiée n'existe pas.
    DirectionInvalide,
    /// Le pigeon spécifié n'existe pas.
    PigeonInvalide,
}

/// Directions possibles
#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq, PartialOrd, Ord)]
pub enum Direction {
    /// Sens positif pour les lignes
    Nord,
    /// Sens négatif pour les lignes
    Sud,
    /// Sens positif pour les colonnes
    Est,
    /// Sens négatif pour les colonnes
    Ouest,
    /// Sens positif pour le niveau
    Haut,
    /// Sens négatif pour le niveau
    Bas,
}

/// Type de l'élément présent sur une case
#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq, PartialOrd, Ord)]
pub enum TypeCase {
    /// Absence d'élément
    Gazon,
    /// Obstacle impossible à traverser
    Buisson,
    /// Élément pouvant être ouvert ou fermé. Une barrière fermée est
    /// infranchissable alors qu'une barrière ouverte est analogue à une case
    /// vide
    Barriere,
    /// Élément traversable permettant à la troupe de déposer son inventaire en
    /// échange de points
    Nid,
    /// Élément traversable générant de manière périodique des miches de pain
    Papy,
    /// Interface entre le niveau principal est le niveau souterrain
    Trou,
    /// Bloc du souterrain ayant été creusé
    Tunnel,
    /// Bloc du souterrain n'ayant pas encore été creusé
    Terre,
}

/// État d'une barrière, soit ouvert, soit fermé, soit non-applicable
#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq, PartialOrd, Ord)]
pub enum EtatBarriere {
    /// La barrière est ouverte
    Ouverte,
    /// La barrière est fermée
    Fermee,
    /// L'élément dont on requiert l'état n'est pas une barrière
    PasDeBarriere,
}

/// Joueur auquel appartient un nid
#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq, PartialOrd, Ord)]
pub enum EtatNid {
    /// Le nid n'a pas été attribué
    Libre,
    /// Joueur 0
    Joueur0,
    /// Joueur 1
    Joueur1,
    /// L'élément dont on requiert l'état n'est pas un nid
    PasDeNid,
}

/// Type de pigeon de debug
#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq, PartialOrd, Ord)]
pub enum PigeonDebug {
    /// Aucun pigeon, enlève le pigeon présent
    PasDePigeon,
    /// Pigeon bleu
    PigeonBleu,
    /// Pigeon jaune
    PigeonJaune,
    /// Pigeon rouge
    PigeonRouge,
}

/// Types d'actions
#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq, PartialOrd, Ord)]
pub enum TypeAction {
    /// Action ``avancer``
    ActionAvancer,
    /// Action ``grandir``
    ActionGrandir,
    /// Action ``construire buisson``
    ActionConstruire,
    /// Action ``creuser tunnel``
    ActionCreuser,
}

/// Position dans la carte, donnée par trois coordonnées
/// ### Fields
/// - Abscisse
/// - Ordonnée
/// - Niveau
pub type Position = (i32, i32, i32);

/// Une troupe, composée de la maman canard et de ses canetons
#[derive(Clone, Debug, Eq, Hash, Ord, PartialEq, PartialOrd)]
pub struct Troupe {
    /// Position de la maman canard
    pub maman: Position,
    /// Position des différents canards de la troupe, incluant la maman en
    /// première position
    pub canards: Vec<Position>,
    /// Taille de la troupe
    pub taille: i32,
    /// Direction de la troupe
    pub dir: Direction,
    /// Nombre de pains de la troupe
    pub inventaire: i32,
    /// Nombre de points d'action de la troupe
    pub pts_action: i32,
    /// Identifiant de la troupe
    pub id: i32,
}

/// Élément constituant le parc
#[derive(Clone, Copy, Debug, Eq, Hash, Ord, PartialEq, PartialOrd)]
pub struct EtatCase {
    /// Position de la case. Le niveau vaut nécessairement 0
    pub pos: Position,
    /// Type de la case
    pub contenu: TypeCase,
    /// La case est constructible
    pub est_constructible: bool,
    /// Nombre de pains contenus sur la case
    pub nb_pains: i32,
}

/// Action représentée dans l'historique
#[derive(Clone, Copy, Debug, Eq, Hash, Ord, PartialEq, PartialOrd)]
pub struct ActionHist {
    /// Type de l'action
    pub action_type: TypeAction,
    /// Identifiant de la troupe
    pub troupe_id: i32,
    /// Direction de l'action
    pub action_dir: Direction,
    /// Position de l'action
    pub action_pos: Position,
}


/// La troupe avance d'une case vers une direction donnée
///
/// ### Parameters
///  - `id`: Identifiant de la troupe à avancer
///  - `dir`: Direction vers laquelle avancer
pub fn avancer(id: i32, dir: Direction) -> Erreur {
    unsafe {
        let id = id.to_c();
        let dir = dir.to_c();
        ffi::avancer(id, dir).to_rust()
    }
}

/// La troupe grandit
///
/// ### Parameters
///  - `id`: Identifiant de la troupe à faire grandir
pub fn grandir(id: i32) -> Erreur {
    unsafe {
        let id = id.to_c();
        ffi::grandir(id).to_rust()
    }
}

/// Construit un buisson à la position donnée
///
/// ### Parameters
///  - `pos`: Position où construire le buisson
pub fn construire_buisson(pos: Position) -> Erreur {
    unsafe {
        let pos = pos.to_c();
        ffi::construire_buisson(pos).to_rust()
    }
}

/// Creuse un tunnel à la position donnée
///
/// ### Parameters
///  - `pos`: Position de la case à creuser
pub fn creuser_tunnel(pos: Position) -> Erreur {
    unsafe {
        let pos = pos.to_c();
        ffi::creuser_tunnel(pos).to_rust()
    }
}

/// Renvoie les informations concernant une case
///
/// ### Parameters
///  - `pos`: Position de la case
pub fn info_case(pos: Position) -> EtatCase {
    unsafe {
        let pos = pos.to_c();
        ffi::info_case(pos).to_rust()
    }
}

/// Renvoie les informations d'état d'une barrière
///
/// ### Parameters
///  - `pos`: Position de la barrière
pub fn info_barriere(pos: Position) -> EtatBarriere {
    unsafe {
        let pos = pos.to_c();
        ffi::info_barriere(pos).to_rust()
    }
}

/// Renvoie les informations d'état d'un nid
///
/// ### Parameters
///  - `pos`: Position du nid
pub fn info_nid(pos: Position) -> EtatNid {
    unsafe {
        let pos = pos.to_c();
        ffi::info_nid(pos).to_rust()
    }
}

/// Renvoie le nombre de tours restants avant qu'un papy dépose une miche de
/// pain. Retourne -1 si aucun papy ne se trouve à la position demandée
///
/// ### Parameters
///  - `pos`: Position du papy
pub fn papy_tours_restants(pos: Position) -> i32 {
    unsafe {
        let pos = pos.to_c();
        ffi::papy_tours_restants(pos).to_rust()
    }
}

/// Renvoie les troupes d'un joueur. Si le joueur est invalide, tous les champs
/// valent -1.
///
/// ### Parameters
///  - `id_joueur`: Numéro du joueur concerné
pub fn troupes_joueur(id_joueur: i32) -> Vec<Troupe> {
    unsafe {
        let id_joueur = id_joueur.to_c();
        ffi::troupes_joueur(id_joueur).to_rust()
    }
}

/// Renvoie la position des pains récupérables
pub fn pains() -> Vec<Position> {
    unsafe {
        ffi::pains().to_rust()
    }
}

/// Pose un pigeon de debug sur la case indiquée
///
/// ### Parameters
///  - `pos`: Case où poser le pigeon
///  - `pigeon`: Pigeon à afficher sur la case
pub fn debug_poser_pigeon(pos: Position, pigeon: PigeonDebug) -> Erreur {
    unsafe {
        let pos = pos.to_c();
        let pigeon = pigeon.to_c();
        ffi::debug_poser_pigeon(pos, pigeon).to_rust()
    }
}

/// Renvoie la liste des actions effectuées par l'adversaire durant son tour,
/// dans l'ordre chronologique. Les actions de débug n'apparaissent pas dans
/// cette liste.
pub fn historique() -> Vec<ActionHist> {
    unsafe {
        ffi::historique().to_rust()
    }
}

/// Renvoie le gain en score que le nombre de pains passé en entrée
/// rapporterait s'ils étaient tous déposés d'un coup dans un nid
///
/// ### Parameters
///  - `nb_pains`: Nombre de miches de pain déposées
pub fn gain(nb_pains: i32) -> i32 {
    unsafe {
        let nb_pains = nb_pains.to_c();
        ffi::gain(nb_pains).to_rust()
    }
}

/// Renvoie la taille de l'inventaire d'une troupe de taille donnée
///
/// ### Parameters
///  - `taille`: Taille de la troupe
pub fn inventaire(taille: i32) -> i32 {
    unsafe {
        let taille = taille.to_c();
        ffi::inventaire(taille).to_rust()
    }
}

/// Trouve un plus court chemin ouvert entre deux positions. Renvoie une liste
/// vide si les deux positions sont égales ou si aucun chemin n'existe.
///
/// ### Parameters
///  - `depart`: Position de départ
///  - `arrivee`: Position d'arrivée
pub fn trouver_chemin(depart: Position, arrivee: Position) -> Vec<Direction> {
    unsafe {
        let depart = depart.to_c();
        let arrivee = arrivee.to_c();
        ffi::trouver_chemin(depart, arrivee).to_rust()
    }
}

/// Renvoie votre numéro de joueur.
pub fn moi() -> i32 {
    unsafe {
        ffi::moi().to_rust()
    }
}

/// Renvoie le numéro du joueur adverse.
pub fn adversaire() -> i32 {
    unsafe {
        ffi::adversaire().to_rust()
    }
}

/// Renvoie le score du joueur `id_joueur`. Renvoie -1 si le joueur est
/// invalide.
///
/// ### Parameters
///  - `id_joueur`: Numéro du joueur concerné
pub fn score(id_joueur: i32) -> i32 {
    unsafe {
        let id_joueur = id_joueur.to_c();
        ffi::score(id_joueur).to_rust()
    }
}

/// Annule la dernière action. Renvoie faux quand il n'y a pas d'action à
/// annuler ce tour-ci
pub fn annuler() -> bool {
    unsafe {
        ffi::annuler().to_rust()
    }
}

/// Retourne le numéro du tour actuel.
pub fn tour_actuel() -> i32 {
    unsafe {
        ffi::tour_actuel().to_rust()
    }
}

/// Affiche le contenu d'une valeur de type erreur
///
/// ### Parameters
///  - `v`: The value to display
pub fn afficher_erreur(v: Erreur) {
    unsafe {
        let v = v.to_c();
        ffi::afficher_erreur(v).to_rust()
    }
}

/// Affiche le contenu d'une valeur de type direction
///
/// ### Parameters
///  - `v`: The value to display
pub fn afficher_direction(v: Direction) {
    unsafe {
        let v = v.to_c();
        ffi::afficher_direction(v).to_rust()
    }
}

/// Affiche le contenu d'une valeur de type type_case
///
/// ### Parameters
///  - `v`: The value to display
pub fn afficher_type_case(v: TypeCase) {
    unsafe {
        let v = v.to_c();
        ffi::afficher_type_case(v).to_rust()
    }
}

/// Affiche le contenu d'une valeur de type etat_barriere
///
/// ### Parameters
///  - `v`: The value to display
pub fn afficher_etat_barriere(v: EtatBarriere) {
    unsafe {
        let v = v.to_c();
        ffi::afficher_etat_barriere(v).to_rust()
    }
}

/// Affiche le contenu d'une valeur de type etat_nid
///
/// ### Parameters
///  - `v`: The value to display
pub fn afficher_etat_nid(v: EtatNid) {
    unsafe {
        let v = v.to_c();
        ffi::afficher_etat_nid(v).to_rust()
    }
}

/// Affiche le contenu d'une valeur de type pigeon_debug
///
/// ### Parameters
///  - `v`: The value to display
pub fn afficher_pigeon_debug(v: PigeonDebug) {
    unsafe {
        let v = v.to_c();
        ffi::afficher_pigeon_debug(v).to_rust()
    }
}

/// Affiche le contenu d'une valeur de type type_action
///
/// ### Parameters
///  - `v`: The value to display
pub fn afficher_type_action(v: TypeAction) {
    unsafe {
        let v = v.to_c();
        ffi::afficher_type_action(v).to_rust()
    }
}

/// Affiche le contenu d'une valeur de type position
///
/// ### Parameters
///  - `v`: The value to display
pub fn afficher_position(v: Position) {
    unsafe {
        let v = v.to_c();
        ffi::afficher_position(v).to_rust()
    }
}

/// Affiche le contenu d'une valeur de type troupe
///
/// ### Parameters
///  - `v`: The value to display
pub fn afficher_troupe(v: &Troupe) {
    unsafe {
        let v = UnsafeCell::new(v.to_c());
        ffi::afficher_troupe(v.get().read()).to_rust()
    }
}

/// Affiche le contenu d'une valeur de type etat_case
///
/// ### Parameters
///  - `v`: The value to display
pub fn afficher_etat_case(v: &EtatCase) {
    unsafe {
        let v = UnsafeCell::new(v.to_c());
        ffi::afficher_etat_case(v.get().read()).to_rust()
    }
}

/// Affiche le contenu d'une valeur de type action_hist
///
/// ### Parameters
///  - `v`: The value to display
pub fn afficher_action_hist(v: &ActionHist) {
    unsafe {
        let v = UnsafeCell::new(v.to_c());
        ffi::afficher_action_hist(v.get().read()).to_rust()
    }
}
