// SPDX-License-Identifier: GPL-2.0-or-later
// Copyright (c) 2012-2020 Association Prologin <association@prologin.org>

//! Types and conversions for the C interface
//!
//! Please use the tools defined in `api.rs` to interact with the API for
//! prologin2022.

#![allow(clippy::unit_arg)]
#![allow(clippy::unused_unit)]

use crate::api;

use std::borrow::Borrow;
use std::ffi::{CStr, CString};
use std::{mem::{drop, size_of}, ptr, slice};
use std::os::raw::{c_char, c_double, c_int, c_void};

#[allow(non_camel_case_types)]
pub type c_bool = bool;

/// Stechec2-specific array type.
#[repr(C)]
pub struct Array<T> {
    ptr: *mut T,
    len: usize,
}

impl<T> Drop for Array<T> {
    fn drop(&mut self) {
        unsafe {
            slice::from_raw_parts_mut(self.ptr, self.len)
                .iter_mut()
                .for_each(drop);
            free(self.ptr as _);
        }
    }
}

/// Represents an owned C string that was created by a foreign function using
/// malloc. This means that this string must be deallocated using free.
#[repr(C)]
pub struct RawString {
    ptr: *mut c_char
}

impl Drop for RawString {
    fn drop(&mut self) {
        unsafe {
            free(self.ptr as _);
        }
    }
}

// Enums

#[repr(C)]
#[derive(Clone, Copy)]
pub enum Erreur {
    Ok,
    TroupeInvalide,
    HorsTour,
    MouvementsInsuffisants,
    TropGrandi,
    TropCreuse,
    NonCreusable,
    NonConstructible,
    ScoreInsuffisant,
    PositionInvalide,
    DirectionInvalide,
    PigeonInvalide,
}

#[repr(C)]
#[derive(Clone, Copy)]
pub enum Direction {
    Nord,
    Sud,
    Est,
    Ouest,
    Haut,
    Bas,
}

#[repr(C)]
#[derive(Clone, Copy)]
pub enum TypeCase {
    Gazon,
    Buisson,
    Barriere,
    Nid,
    Papy,
    Trou,
    Tunnel,
    Terre,
}

#[repr(C)]
#[derive(Clone, Copy)]
pub enum EtatBarriere {
    Ouverte,
    Fermee,
    PasDeBarriere,
}

#[repr(C)]
#[derive(Clone, Copy)]
pub enum EtatNid {
    Libre,
    Joueur0,
    Joueur1,
    PasDeNid,
}

#[repr(C)]
#[derive(Clone, Copy)]
pub enum PigeonDebug {
    PasDePigeon,
    PigeonBleu,
    PigeonJaune,
    PigeonRouge,
}

#[repr(C)]
#[derive(Clone, Copy)]
pub enum TypeAction {
    ActionAvancer,
    ActionGrandir,
    ActionConstruire,
    ActionCreuser,
}

// Structures

#[repr(C)]
pub struct Position {
    colonne: c_int,
    ligne: c_int,
    niveau: c_int,
}

#[repr(C)]
pub struct Troupe {
    maman: Position,
    canards: Array<Position>,
    taille: c_int,
    dir: Direction,
    inventaire: c_int,
    pts_action: c_int,
    id: c_int,
}

#[repr(C)]
pub struct EtatCase {
    pos: Position,
    contenu: TypeCase,
    est_constructible: c_bool,
    nb_pains: c_int,
}

#[repr(C)]
pub struct ActionHist {
    action_type: TypeAction,
    troupe_id: c_int,
    action_dir: Direction,
    action_pos: Position,
}

// Conversion traits

pub trait CToRust<T> {
    /// Convert from a C-compatible type.
    ///
    /// As there can't be a clear ownership through the ffi, you need to make
    /// sure that foreign code assumes that you will drop provided values.
    unsafe fn to_rust(self) -> T;
}

pub trait RustToC<T> {
    /// Convert to a C-compatible type.
    ///
    /// As there can't be a clear ownership through the ffi, you need to make
    /// sure that foreign code assumes that you will drop provided values.
    unsafe fn to_c(&self) -> T;
}

// Conversions for bool

impl CToRust<bool> for c_bool {
    unsafe fn to_rust(self) -> bool {
        self
    }
}

impl RustToC<c_bool> for bool {
    unsafe fn to_c(&self) -> c_bool {
        *self
    }
}

// Conversions for double

impl CToRust<f64> for c_double {
    unsafe fn to_rust(self) -> f64 {
        self
    }
}

impl RustToC<c_double> for f64 {
    unsafe fn to_c(&self) -> c_double {
        *self
    }
}

// Conversions for int

impl CToRust<i32> for c_int {
    unsafe fn to_rust(self) -> i32 {
        self
    }
}

impl RustToC<c_int> for i32 {
    unsafe fn to_c(&self) -> c_int {
        *self
    }
}

// Conversions for void

impl CToRust<()> for () {
    unsafe fn to_rust(self) -> () {
        self
    }
}

impl RustToC<()> for () {
    unsafe fn to_c(&self) -> () {
        *self
    }
}

// Conversions for string

impl CToRust<String> for RawString {
    unsafe fn to_rust(self) -> String {
        CStr::from_ptr(self.ptr)
            .to_owned()
            .into_string()
            .expect("API provided invalid UTF-8")
    }
}

impl<S> RustToC<RawString> for S
where
    S: AsRef<str>,
{
    unsafe fn to_c(&self) -> RawString {
        let c_string = CString::new(self.as_ref().to_string())
            .expect("string provided to the API contains a null character");
        let len = c_string.as_bytes_with_nul().len();

        let ptr = malloc(len * size_of::<c_char>()) as *mut c_char;
        c_string.as_c_str().as_ptr().copy_to(ptr, len);
        RawString { ptr }
    }
}

// Conversions for array

pub unsafe fn array_of_borrow_to_c<T, U, V>(data: &[T]) -> Array<V>
where
    T: Borrow<U>,
    U: RustToC<V>,
{
    let ptr = malloc(data.len() * size_of::<V>()) as *mut V;

    for (i, item) in data.iter().enumerate() {
        ptr::write(ptr.add(i), item.borrow().to_c());
    }

    Array { ptr, len: data.len() }
}

impl<T, U> CToRust<Vec<U>> for Array<T>
where
    T: CToRust<U>,
{
    unsafe fn to_rust(self) -> Vec<U> {
        (0..self.len)
            .map(|i| self.ptr.add(i).read())
            .map(|item| item.to_rust())
            .collect()
    }
}

impl<T, U> RustToC<Array<U>> for [T]
where
    T: RustToC<U>,
{
    unsafe fn to_c(&self) -> Array<U> {
        array_of_borrow_to_c(self)
    }
}

impl<T, U> RustToC<Array<U>> for Vec<T>
where
    T: RustToC<U>,
{
    unsafe fn to_c(&self) -> Array<U> {
        self[..].to_c()
    }
}

// Conversions for erreur

impl CToRust<api::Erreur> for Erreur {
    unsafe fn to_rust(self) -> api::Erreur {
        match self {
            Erreur::Ok => api::Erreur::Ok,
            Erreur::TroupeInvalide => api::Erreur::TroupeInvalide,
            Erreur::HorsTour => api::Erreur::HorsTour,
            Erreur::MouvementsInsuffisants => api::Erreur::MouvementsInsuffisants,
            Erreur::TropGrandi => api::Erreur::TropGrandi,
            Erreur::TropCreuse => api::Erreur::TropCreuse,
            Erreur::NonCreusable => api::Erreur::NonCreusable,
            Erreur::NonConstructible => api::Erreur::NonConstructible,
            Erreur::ScoreInsuffisant => api::Erreur::ScoreInsuffisant,
            Erreur::PositionInvalide => api::Erreur::PositionInvalide,
            Erreur::DirectionInvalide => api::Erreur::DirectionInvalide,
            Erreur::PigeonInvalide => api::Erreur::PigeonInvalide,
        }
    }
}

impl RustToC<Erreur> for api::Erreur {
    unsafe fn to_c(&self) -> Erreur {
        match self {
            api::Erreur::Ok => Erreur::Ok,
            api::Erreur::TroupeInvalide => Erreur::TroupeInvalide,
            api::Erreur::HorsTour => Erreur::HorsTour,
            api::Erreur::MouvementsInsuffisants => Erreur::MouvementsInsuffisants,
            api::Erreur::TropGrandi => Erreur::TropGrandi,
            api::Erreur::TropCreuse => Erreur::TropCreuse,
            api::Erreur::NonCreusable => Erreur::NonCreusable,
            api::Erreur::NonConstructible => Erreur::NonConstructible,
            api::Erreur::ScoreInsuffisant => Erreur::ScoreInsuffisant,
            api::Erreur::PositionInvalide => Erreur::PositionInvalide,
            api::Erreur::DirectionInvalide => Erreur::DirectionInvalide,
            api::Erreur::PigeonInvalide => Erreur::PigeonInvalide,
        }
    }
}

// Conversions for direction

impl CToRust<api::Direction> for Direction {
    unsafe fn to_rust(self) -> api::Direction {
        match self {
            Direction::Nord => api::Direction::Nord,
            Direction::Sud => api::Direction::Sud,
            Direction::Est => api::Direction::Est,
            Direction::Ouest => api::Direction::Ouest,
            Direction::Haut => api::Direction::Haut,
            Direction::Bas => api::Direction::Bas,
        }
    }
}

impl RustToC<Direction> for api::Direction {
    unsafe fn to_c(&self) -> Direction {
        match self {
            api::Direction::Nord => Direction::Nord,
            api::Direction::Sud => Direction::Sud,
            api::Direction::Est => Direction::Est,
            api::Direction::Ouest => Direction::Ouest,
            api::Direction::Haut => Direction::Haut,
            api::Direction::Bas => Direction::Bas,
        }
    }
}

// Conversions for type_case

impl CToRust<api::TypeCase> for TypeCase {
    unsafe fn to_rust(self) -> api::TypeCase {
        match self {
            TypeCase::Gazon => api::TypeCase::Gazon,
            TypeCase::Buisson => api::TypeCase::Buisson,
            TypeCase::Barriere => api::TypeCase::Barriere,
            TypeCase::Nid => api::TypeCase::Nid,
            TypeCase::Papy => api::TypeCase::Papy,
            TypeCase::Trou => api::TypeCase::Trou,
            TypeCase::Tunnel => api::TypeCase::Tunnel,
            TypeCase::Terre => api::TypeCase::Terre,
        }
    }
}

impl RustToC<TypeCase> for api::TypeCase {
    unsafe fn to_c(&self) -> TypeCase {
        match self {
            api::TypeCase::Gazon => TypeCase::Gazon,
            api::TypeCase::Buisson => TypeCase::Buisson,
            api::TypeCase::Barriere => TypeCase::Barriere,
            api::TypeCase::Nid => TypeCase::Nid,
            api::TypeCase::Papy => TypeCase::Papy,
            api::TypeCase::Trou => TypeCase::Trou,
            api::TypeCase::Tunnel => TypeCase::Tunnel,
            api::TypeCase::Terre => TypeCase::Terre,
        }
    }
}

// Conversions for etat_barriere

impl CToRust<api::EtatBarriere> for EtatBarriere {
    unsafe fn to_rust(self) -> api::EtatBarriere {
        match self {
            EtatBarriere::Ouverte => api::EtatBarriere::Ouverte,
            EtatBarriere::Fermee => api::EtatBarriere::Fermee,
            EtatBarriere::PasDeBarriere => api::EtatBarriere::PasDeBarriere,
        }
    }
}

impl RustToC<EtatBarriere> for api::EtatBarriere {
    unsafe fn to_c(&self) -> EtatBarriere {
        match self {
            api::EtatBarriere::Ouverte => EtatBarriere::Ouverte,
            api::EtatBarriere::Fermee => EtatBarriere::Fermee,
            api::EtatBarriere::PasDeBarriere => EtatBarriere::PasDeBarriere,
        }
    }
}

// Conversions for etat_nid

impl CToRust<api::EtatNid> for EtatNid {
    unsafe fn to_rust(self) -> api::EtatNid {
        match self {
            EtatNid::Libre => api::EtatNid::Libre,
            EtatNid::Joueur0 => api::EtatNid::Joueur0,
            EtatNid::Joueur1 => api::EtatNid::Joueur1,
            EtatNid::PasDeNid => api::EtatNid::PasDeNid,
        }
    }
}

impl RustToC<EtatNid> for api::EtatNid {
    unsafe fn to_c(&self) -> EtatNid {
        match self {
            api::EtatNid::Libre => EtatNid::Libre,
            api::EtatNid::Joueur0 => EtatNid::Joueur0,
            api::EtatNid::Joueur1 => EtatNid::Joueur1,
            api::EtatNid::PasDeNid => EtatNid::PasDeNid,
        }
    }
}

// Conversions for pigeon_debug

impl CToRust<api::PigeonDebug> for PigeonDebug {
    unsafe fn to_rust(self) -> api::PigeonDebug {
        match self {
            PigeonDebug::PasDePigeon => api::PigeonDebug::PasDePigeon,
            PigeonDebug::PigeonBleu => api::PigeonDebug::PigeonBleu,
            PigeonDebug::PigeonJaune => api::PigeonDebug::PigeonJaune,
            PigeonDebug::PigeonRouge => api::PigeonDebug::PigeonRouge,
        }
    }
}

impl RustToC<PigeonDebug> for api::PigeonDebug {
    unsafe fn to_c(&self) -> PigeonDebug {
        match self {
            api::PigeonDebug::PasDePigeon => PigeonDebug::PasDePigeon,
            api::PigeonDebug::PigeonBleu => PigeonDebug::PigeonBleu,
            api::PigeonDebug::PigeonJaune => PigeonDebug::PigeonJaune,
            api::PigeonDebug::PigeonRouge => PigeonDebug::PigeonRouge,
        }
    }
}

// Conversions for type_action

impl CToRust<api::TypeAction> for TypeAction {
    unsafe fn to_rust(self) -> api::TypeAction {
        match self {
            TypeAction::ActionAvancer => api::TypeAction::ActionAvancer,
            TypeAction::ActionGrandir => api::TypeAction::ActionGrandir,
            TypeAction::ActionConstruire => api::TypeAction::ActionConstruire,
            TypeAction::ActionCreuser => api::TypeAction::ActionCreuser,
        }
    }
}

impl RustToC<TypeAction> for api::TypeAction {
    unsafe fn to_c(&self) -> TypeAction {
        match self {
            api::TypeAction::ActionAvancer => TypeAction::ActionAvancer,
            api::TypeAction::ActionGrandir => TypeAction::ActionGrandir,
            api::TypeAction::ActionConstruire => TypeAction::ActionConstruire,
            api::TypeAction::ActionCreuser => TypeAction::ActionCreuser,
        }
    }
}

// Conversions for position

impl CToRust<api::Position> for Position {
    unsafe fn to_rust(self) -> api::Position {
        (self.colonne.to_rust(), self.ligne.to_rust(), self.niveau.to_rust())
    }
}

impl RustToC<Position> for api::Position {
    unsafe fn to_c(&self) -> Position {
        let (colonne, ligne, niveau) = self;

        Position {
            colonne: colonne.to_c(),
            ligne: ligne.to_c(),
            niveau: niveau.to_c(),
        }
    }
}

// Conversions for troupe

impl CToRust<api::Troupe> for Troupe {
    unsafe fn to_rust(self) -> api::Troupe {
        api::Troupe {
            maman: self.maman.to_rust(),
            canards: self.canards.to_rust(),
            taille: self.taille.to_rust(),
            dir: self.dir.to_rust(),
            inventaire: self.inventaire.to_rust(),
            pts_action: self.pts_action.to_rust(),
            id: self.id.to_rust(),
        }
    }
}

impl RustToC<Troupe> for api::Troupe {
    unsafe fn to_c(&self) -> Troupe {
        Troupe {
            maman: self.maman.to_c(),
            canards: self.canards.to_c(),
            taille: self.taille.to_c(),
            dir: self.dir.to_c(),
            inventaire: self.inventaire.to_c(),
            pts_action: self.pts_action.to_c(),
            id: self.id.to_c(),
        }
    }
}

// Conversions for etat_case

impl CToRust<api::EtatCase> for EtatCase {
    unsafe fn to_rust(self) -> api::EtatCase {
        api::EtatCase {
            pos: self.pos.to_rust(),
            contenu: self.contenu.to_rust(),
            est_constructible: self.est_constructible.to_rust(),
            nb_pains: self.nb_pains.to_rust(),
        }
    }
}

impl RustToC<EtatCase> for api::EtatCase {
    unsafe fn to_c(&self) -> EtatCase {
        EtatCase {
            pos: self.pos.to_c(),
            contenu: self.contenu.to_c(),
            est_constructible: self.est_constructible.to_c(),
            nb_pains: self.nb_pains.to_c(),
        }
    }
}

// Conversions for action_hist

impl CToRust<api::ActionHist> for ActionHist {
    unsafe fn to_rust(self) -> api::ActionHist {
        api::ActionHist {
            action_type: self.action_type.to_rust(),
            troupe_id: self.troupe_id.to_rust(),
            action_dir: self.action_dir.to_rust(),
            action_pos: self.action_pos.to_rust(),
        }
    }
}

impl RustToC<ActionHist> for api::ActionHist {
    unsafe fn to_c(&self) -> ActionHist {
        ActionHist {
            action_type: self.action_type.to_c(),
            troupe_id: self.troupe_id.to_c(),
            action_dir: self.action_dir.to_c(),
            action_pos: self.action_pos.to_c(),
        }
    }
}


// Import API functions

extern {
    fn free(ptr: *mut c_void);
    fn malloc(size: usize) -> *mut c_void;

    pub fn avancer(id: c_int, dir: Direction) -> Erreur;
    pub fn grandir(id: c_int) -> Erreur;
    pub fn construire_buisson(pos: Position) -> Erreur;
    pub fn creuser_tunnel(pos: Position) -> Erreur;
    pub fn info_case(pos: Position) -> EtatCase;
    pub fn info_barriere(pos: Position) -> EtatBarriere;
    pub fn info_nid(pos: Position) -> EtatNid;
    pub fn papy_tours_restants(pos: Position) -> c_int;
    pub fn troupes_joueur(id_joueur: c_int) -> Array<Troupe>;
    pub fn pains() -> Array<Position>;
    pub fn debug_poser_pigeon(pos: Position, pigeon: PigeonDebug) -> Erreur;
    pub fn historique() -> Array<ActionHist>;
    pub fn gain(nb_pains: c_int) -> c_int;
    pub fn inventaire(taille: c_int) -> c_int;
    pub fn trouver_chemin(depart: Position, arrivee: Position) -> Array<Direction>;
    pub fn moi() -> c_int;
    pub fn adversaire() -> c_int;
    pub fn score(id_joueur: c_int) -> c_int;
    pub fn annuler() -> c_bool;
    pub fn tour_actuel() -> c_int;
    pub fn afficher_erreur(v: Erreur);
    pub fn afficher_direction(v: Direction);
    pub fn afficher_type_case(v: TypeCase);
    pub fn afficher_etat_barriere(v: EtatBarriere);
    pub fn afficher_etat_nid(v: EtatNid);
    pub fn afficher_pigeon_debug(v: PigeonDebug);
    pub fn afficher_type_action(v: TypeAction);
    pub fn afficher_position(v: Position);
    pub fn afficher_troupe(v: Troupe);
    pub fn afficher_etat_case(v: EtatCase);
    pub fn afficher_action_hist(v: ActionHist);
}

// Export user functions

#[no_mangle]
unsafe extern "C" fn partie_init() {
    crate::partie_init().to_c()
}

#[no_mangle]
unsafe extern "C" fn jouer_tour() {
    crate::jouer_tour().to_c()
}

#[no_mangle]
unsafe extern "C" fn partie_fin() {
    crate::partie_fin().to_c()
}
