#![allow(unused)]
use ffi::{self, ToAPI, ToFFI};
use std::os::raw::{c_float, c_double, c_int, c_void};

// Taille de la mine (longueur et largeur).
pub const TAILLE_MINE: c_int = 31;

// Nombre de tours à jouer avant la fin de la partie.
pub const NB_TOURS: c_int = 100;

// Nombre de points de déplacement par tour par nain (standard).
pub const NB_POINTS_DEPLACEMENT: c_int = 5;

// Nombre de points d'action par tour par nain (standard).
pub const NB_POINTS_ACTION: c_int = 6;

// Nombre de points de vie d'un nain (standard).
pub const VIE_NAIN: c_int = 10;

// Nombre de joueurs.
pub const NB_JOUEURS: c_int = 2;

// Nombre de nains (standard) par joueur.
pub const NB_NAINS: c_int = 6;

// Dégât infligé par un coup de pioche à un nain (standard).
pub const DEGAT_PIOCHE: c_int = 3;

// Valeur cumulée maximale des minerais qu'un nain (standard) peut emporter avec lui.
pub const BUTIN_MAX: c_int = 25;

// Nombre de points de déplacement pour qu'un nain (standard) se déplace d'une case.
pub const COUT_DEPLACEMENT: c_int = 1;

// Nombre de points de déplacement pour qu'un nain (standard) se déplace d'une case lorsqu'il est agrippé.
pub const COUT_ESCALADER: c_int = 2;

// Nombre de points de déplacement pour qu'un nain (standard) se déplace vers une case occupée par une corde lorsqu'il est agrippé.
pub const COUT_ESCALADER_CORDE: c_int = 1;

// Nombre de points d'action pour qu'un nain (standard) mine un bloc.
pub const COUT_MINER: c_int = 6;

// Nombre de points d'action pour qu'un nain (standard) tire sur une corde.
pub const COUT_TIRER: c_int = 1;

// Nombre de points d'action pour qu'un nain (standard) lâche la paroi.
pub const COUT_LACHER: c_int = 0;

// Nombre de points d'action pour qu'un nain (standard) s'agrippe à la paroi.
pub const COUT_AGRIPPER: c_int = 0;

// Types de cases
#[repr(C)]
#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq, PartialOrd, Ord)]
pub enum CaseType {
  /// Case libre, qui peut abriter une corde et des nains (standard)
  Libre, 
  /// Granite (standard), qui peut cacher du minerai
  Granite, 
  /// Obsidienne
  Obsidienne, 
  /// Erreur
  ErreurCase, 
}

// Direction
#[repr(C)]
#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq, PartialOrd, Ord)]
pub enum Direction {
  /// Direction : haut
  Haut, 
  /// Direction : bas
  Bas, 
  /// Direction : gauche
  Gauche, 
  /// Direction : droite
  Droite, 
  /// Erreur
  ErreurDirection, 
}

// Erreurs possibles
#[repr(C)]
#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq, PartialOrd, Ord)]
pub enum Erreur {
  /// L'action s'est effectuée avec succès.
  Ok, 
  /// Votre nain (standard) ne possède pas assez de points d'action pour réaliser cette action.
  PaInsuffisants, 
  /// Votre nain (standard) ne possède pas assez de points de déplacement pour réaliser ce déplacement.
  PmInsuffisants, 
  /// L'action est en dehors des limites de la mine.
  HorsLimites, 
  /// La direction spécifiée n'existe pas, ou vous n'êtes pas autorisé à cibler cette direction pour cette action.
  DirectionInvalide, 
  /// Le nain (standard) spécifié n'existe pas.
  IdNainInvalide, 
  /// La position spécifiée est un mur.
  ObstacleMur, 
  /// La position spécifiée est un nain (standard) adverse.
  ObstacleNain, 
  /// Il y a déjà une corde dans la direction spécifiée.
  ObstacleCorde, 
  /// Il n'y a pas de nain (standard) ni de granite (standard) sur la position spécifiée.
  PasDeCible, 
  /// Le nain (standard) spécifié est mort.
  NainMort, 
  /// Le nain (standard) n'est pas accroché.
  PasAccroche, 
  /// Le nain (standard) est déjà accroché.
  DejaAccroche, 
  /// Il n'y a pas de corde dans la direction spécifiée.
  PasDeCorde, 
  /// Le drapeau spécifié n'existe pas.
  DrapeauInvalide, 
}

// Types d'actions
#[repr(C)]
#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq, PartialOrd, Ord)]
pub enum ActionType {
  /// Action ``deplacer``
  ActionDeplacer, 
  /// Action ``lacher``
  ActionLacher, 
  /// Action ``miner``
  ActionMiner, 
  /// Action ``poser_corde``
  ActionPoserCorde, 
  /// Action ``tirer``
  ActionTirer, 
  /// Action ``agripper``
  ActionAgripper, 
}

// Types de drapeaux de débug
#[repr(C)]
#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq, PartialOrd, Ord)]
pub enum DebugDrapeau {
  /// Aucun drapeau, enlève le drapeau présent
  AucunDrapeau, 
  /// Drapeau bleu
  DrapeauBleu, 
  /// Drapeau vert
  DrapeauVert, 
  /// Drapeau rouge
  DrapeauRouge, 
}

// Position dans la mine, donnée par deux coordonnées.
#[derive(Clone, Debug, PartialEq, PartialOrd)]
pub struct Position {
  /// Coordonnée : ligne
  pub ligne: c_int,
  /// Coordonnée : colonne
  pub colonne: c_int,
}


// Minerai à récolter
#[derive(Clone, Debug, PartialEq, PartialOrd)]
pub struct Minerai {
  /// Nombre de coups de pioches encore nécessaires avant que le bloc de minerais ne casse
  pub resistance: c_int,
  /// Valeur marchande du bloc de minerai
  pub rendement: c_int,
}


// Nain (standard)
#[derive(Clone, Debug, PartialEq, PartialOrd)]
pub struct Nain {
  /// Position actuelle du nain (standard)
  pub pos: Position,
  /// Point(s) de vie restant du nain (standard)
  pub vie: c_int,
  /// Point(s) d'action restant du nain (standard)
  pub pa: c_int,
  /// Point(s) de déplacement restant du nain (standard)
  pub pm: c_int,
  /// Le nain (standard) est accroché à la paroi ou à une corde
  pub accroche: bool,
  /// Valeur marchande totale que le nain (standard) possède
  pub butin: c_int,
}


// Action de déplacement représentée dans l'historique.
#[derive(Clone, Debug, PartialEq, PartialOrd)]
pub struct ActionHist {
  /// Type de l'action
  pub atype: ActionType,
  /// Numéro du nain (standard) concerné par l'action
  pub id_nain: c_int,
  /// Direction visée par le nain (standard) durant le déplacement
  pub dir: Direction,
  /// Sens de l'action, utilisé uniquement pour préciser si l'on doit tirer une corde vers le bas ou vers le haut. Direction doit cibler la droite ou la gauche.
  pub sens: Direction,
}


// Renvoie un chemin entre deux positions de la mine sous la forme d'une suite de directions à emprunter. Ce chemin minimise le nombre de blocs de granite nécessaire à casser. Si la position est invalide ou qu'il n'existe pas de tel chemin, le chemin renvoyé est vide.
pub fn chemin(pos1: &Position, pos2: &Position) -> Vec<Direction>
{
  unsafe { ffi::chemin(pos1.to_ffi(), pos2.to_ffi()).to_api() }
}

// Déplace le nain (standard) ``id_nain`` d'une case dans la direction choisie.
pub fn deplacer(id_nain: &c_int, dir: &Direction) -> Erreur
{
  unsafe { ffi::deplacer(id_nain.to_ffi(), dir.to_ffi()).to_api() }
}

// Le nain (standard) ``id_nain`` lâche la paroi.
pub fn lacher(id_nain: &c_int) -> Erreur
{
  unsafe { ffi::lacher(id_nain.to_ffi()).to_api() }
}

// Le nain (standard) ``id_nain`` s'agrippe à la paroi.
pub fn agripper(id_nain: &c_int) -> Erreur
{
  unsafe { ffi::agripper(id_nain.to_ffi()).to_api() }
}

// Le nain (standard) ``id_nain`` mine le bloc ou le nain (standard) dans la direction indiquée.
pub fn miner(id_nain: &c_int, dir: &Direction) -> Erreur
{
  unsafe { ffi::miner(id_nain.to_ffi(), dir.to_ffi()).to_api() }
}

// Le nain (standard) ``id_nain`` tire la corde dans le sens donné (HAUT ou BAS).
pub fn tirer(id_nain: &c_int, dir_corde: &Direction, sens: &Direction) -> Erreur
{
  unsafe { ffi::tirer(id_nain.to_ffi(), dir_corde.to_ffi(), sens.to_ffi()).to_api() }
}

// Le nain (standard) ``id_nain`` pose une corde dans la direction indiquée.
pub fn poser_corde(id_nain: &c_int, dir: &Direction) -> Erreur
{
  unsafe { ffi::poser_corde(id_nain.to_ffi(), dir.to_ffi()).to_api() }
}

// Affiche le drapeau spécifié sur la case indiquée.
pub fn debug_afficher_drapeau(pos: &Position, drapeau: &DebugDrapeau) -> Erreur
{
  unsafe { ffi::debug_afficher_drapeau(pos.to_ffi(), drapeau.to_ffi()).to_api() }
}

// Renvoie le type d'une case donnée.
pub fn type_case(pos: &Position) -> CaseType
{
  unsafe { ffi::type_case(pos.to_ffi()).to_api() }
}

// Renvoie la liste de toutes les positions occupées par une corde dans la mine.
pub fn liste_cordes() -> Vec<Position>
{
  unsafe { ffi::liste_cordes().to_api() }
}

// Indique si une corde se trouve sur une case donnée.
pub fn corde_sur_case(pos: &Position) -> bool
{
  unsafe { ffi::corde_sur_case(pos.to_ffi()).to_api() }
}

// Renvoie le numéro du joueur à qui appartienent les nains (standard) sur la case indiquée. Renvoie -1 s'il n'y a pas de nain (standard) ou si la position est invalide.
pub fn nain_sur_case(pos: &Position) -> c_int
{
  unsafe { ffi::nain_sur_case(pos.to_ffi()).to_api() }
}

// Renvoie la description du nain (standard) désigné par le numéro ``id_nain`` appartenant au joueur ``id_joueur``. Si le nain (standard)  n'est pas présent sur la carte, tous les membres de la structure ``nain`` renvoyée sont initialisés à -1 (et le champ ``accroche`` à `false`).
pub fn info_nain(id_joueur: &c_int, id_nain: &c_int) -> Nain
{
  unsafe { ffi::info_nain(id_joueur.to_ffi(), id_nain.to_ffi()).to_api() }
}

// Renvoie la liste de tous les minerais dans la mine.
pub fn liste_minerais() -> Vec<Position>
{
  unsafe { ffi::liste_minerais().to_api() }
}

// Renvoie la description d'un minerai en fonction d'une position donnée. Si le minerai n'est pas présent sur la carte, ou si la position est invalide, tous les membres de la structure ``minerai`` renvoyée sont initialisés à -1.
pub fn info_minerai(pos: &Position) -> Minerai
{
  unsafe { ffi::info_minerai(pos.to_ffi()).to_api() }
}

// Renvoie le nombre de points de déplacement que coûterai le déplacement d'un nain (standard) dans une direction donnée. Renvoie -1 si le déplacement n'est pas possible.
pub fn cout_de_deplacement(id_nain: &c_int, dir: &Direction) -> c_int
{
  unsafe { ffi::cout_de_deplacement(id_nain.to_ffi(), dir.to_ffi()).to_api() }
}

// Renvoie la position de la taverne appartenant au joueur ``id_joueur``. Si le joueur n'existe pas, renvoie la position (-1, -1).
pub fn position_taverne(id_joueur: &c_int) -> Position
{
  unsafe { ffi::position_taverne(id_joueur.to_ffi()).to_api() }
}

// Renvoie la liste des actions effectuées par l’adversaire durant son tour, dans l'ordre chronologique. Les actions de débug n'apparaissent pas dans cette liste.
pub fn historique() -> Vec<ActionHist>
{
  unsafe { ffi::historique().to_api() }
}

// Renvoie le score du joueur ``id_joueur``. Renvoie -1 si le joueur est invalide.
pub fn score(id_joueur: &c_int) -> c_int
{
  unsafe { ffi::score(id_joueur.to_ffi()).to_api() }
}

// Renvoie votre numéro de joueur.
pub fn moi() -> c_int
{
  unsafe { ffi::moi().to_api() }
}

// Renvoie le numéro de joueur de votre adversaire.
pub fn adversaire() -> c_int
{
  unsafe { ffi::adversaire().to_api() }
}

// Annule la dernière action. Renvoie faux quand il n'y a pas d'action à annuler ce tour ci.
pub fn annuler() -> bool
{
  unsafe { ffi::annuler().to_api() }
}

// Retourne le numéro du tour actuel.
pub fn tour_actuel() -> c_int
{
  unsafe { ffi::tour_actuel().to_api() }
}

// Affiche le contenu d'une valeur de type case_type
pub fn afficher_case_type(v: &CaseType)
{
  unsafe { ffi::afficher_case_type(v.to_ffi()).to_api() }
}

// Affiche le contenu d'une valeur de type direction
pub fn afficher_direction(v: &Direction)
{
  unsafe { ffi::afficher_direction(v.to_ffi()).to_api() }
}

// Affiche le contenu d'une valeur de type erreur
pub fn afficher_erreur(v: &Erreur)
{
  unsafe { ffi::afficher_erreur(v.to_ffi()).to_api() }
}

// Affiche le contenu d'une valeur de type action_type
pub fn afficher_action_type(v: &ActionType)
{
  unsafe { ffi::afficher_action_type(v.to_ffi()).to_api() }
}

// Affiche le contenu d'une valeur de type debug_drapeau
pub fn afficher_debug_drapeau(v: &DebugDrapeau)
{
  unsafe { ffi::afficher_debug_drapeau(v.to_ffi()).to_api() }
}

// Affiche le contenu d'une valeur de type position
pub fn afficher_position(v: &Position)
{
  unsafe { ffi::afficher_position(v.to_ffi()).to_api() }
}

// Affiche le contenu d'une valeur de type minerai
pub fn afficher_minerai(v: &Minerai)
{
  unsafe { ffi::afficher_minerai(v.to_ffi()).to_api() }
}

// Affiche le contenu d'une valeur de type nain
pub fn afficher_nain(v: &Nain)
{
  unsafe { ffi::afficher_nain(v.to_ffi()).to_api() }
}

// Affiche le contenu d'une valeur de type action_hist
pub fn afficher_action_hist(v: &ActionHist)
{
  unsafe { ffi::afficher_action_hist(v.to_ffi()).to_api() }
}

// Fonction appelée au début de la partie.
#[no_mangle]
pub unsafe extern "C" fn partie_init()
{
  super::partie_init().to_ffi()
}

// Fonction appelée à chaque tour.
#[no_mangle]
pub unsafe extern "C" fn jouer_tour()
{
  super::jouer_tour().to_ffi()
}

// Fonction appelée à la fin de la partie.
#[no_mangle]
pub unsafe extern "C" fn partie_fin()
{
  super::partie_fin().to_ffi()
}

