#include "prologin.hh"
#include <iostream>
#include <vector>
#include <queue>
#include <set>
#include <map>
#include <algorithm>
#include <functional>
#include <tuple>
#include <limits>
#include <random>
#include <chrono>
#include <cassert>

//#define DEBUG

using namespace std;

template<typename T>
using v = vector<T>;

struct dep {
    position from;
    position to ;
    direction dir;
    action_type type;
};

bool operator<(const dep& d1, const dep& d2) {
    return d1.from < d2.from && d1.to < d2.to && d1.dir < d2.dir && d1.type < d2.type;
}

position operator+(const position& a, const position& b) {
    return position{a.ligne + b.ligne, a.colonne + b.colonne};
}

ostream& operator<<(ostream& out, const position& p) {
    out << '(' << p.ligne << ' ' << p.colonne << ')';
    return out;
}

ostream& operator<<(ostream& out, const dep& d) {
    out << d.from << ' ' << d.to << ' ' << d.dir << ' ' << d.type;
    return out;
}

const int INF = numeric_limits<int>::max();
const v<direction> dirs = {NORD, OUEST, SUD, EST};
const v<position> dep_dirs = {{-1, 0}, {0, -1}, {1, 0}, {0, 1}};
const position OUT = {-1, -1};

v<position> assignments(NB_AGENTS, OUT);
map<position, set<int>> targets;

void remove_assignment(int id_agent, position pos) {
    // Remove previous if it exists
    if(targets.count(assignments[id_agent])) {
        targets[assignments[id_agent]].erase(id_agent);
        assignments[id_agent] = OUT;
    }
}

void set_assignment(int id_agent, position pos) {
    remove_assignment(id_agent, pos);
    // Add if it does not exist
    assignments[id_agent] = pos;
    targets[pos].insert(id_agent);
}

bool can_go(position pos) {
    return type_case(pos) == LIBRE;
}

tuple<v<v<int>>, v<v<dep>>> min_move(position init) {
    v<v<int>> dists(TAILLE_BANQUISE, v<int>(TAILLE_BANQUISE, INF));
    v<v<dep>> parents(TAILLE_BANQUISE, v<dep>(TAILLE_BANQUISE));

    priority_queue<pair<int, dep>, v<pair<int, dep>>, greater<pair<int, dep>>> pq;
    pq.push({0, {OUT, init, NORD, ACTION_DEPLACER}});
    while(!pq.empty()) {
        pair<int, dep> p = pq.top(); pq.pop();
        dep& d = p.second;
        position& to = d.to;
        int dist = p.first;
        if (dist >= dists[to.ligne][to.colonne])
            continue;
        dists[to.ligne][to.colonne] = dist;
        parents[to.ligne][to.colonne] = d;
        for(int i = 0; i < (int) dirs.size(); ++i) {
            position dest = to + dep_dirs[i];
            position behind = dest + dep_dirs[i];
            int new_dist = dist + COUT_DEPLACEMENT;
            if (can_go(dest) && agent_sur_case(dest) == -1 && agent_sur_case(behind) != moi()) { // Do not let ennemy on the middle
                if(new_dist < dists[dest.ligne][dest.colonne])
                    pq.push({new_dist, {to, dest, dirs[i], ACTION_DEPLACER}});
            }
        }
        for(int i = 0; i < (int) dirs.size(); ++i) {
            position dest = to + dep_dirs[i];
            position behind = dest + dep_dirs[i];
            int new_dist = dist; // No cost, push is worth it
            if (can_go(dest) && alien_sur_case(dest) && agent_sur_case(dest) == adversaire() &&
                    can_go(behind) && agent_sur_case(behind) == -1) {
                if(new_dist < dists[dest.ligne][dest.colonne])
                    pq.push({new_dist, {to, dest, dirs[i], ACTION_POUSSER}});
            }
        }
        for(int i = 0; i < (int) dirs.size(); ++i) {
            position dest = to + dep_dirs[i];
            if(can_go(dest)) {
                int new_dist = dist + COUT_GLISSADE;
                while(can_go(dest + dep_dirs[i]) && agent_sur_case(dest + dep_dirs[i]) == -1) {
                    dest = dest + dep_dirs[i];
                }
                if(new_dist < dists[dest.ligne][dest.colonne])
                    pq.push({new_dist, {to, dest, dirs[i], ACTION_GLISSER}});
            }
        }
    }
    return {dists, parents};
}

void print_vvi(const v<v<int>>& A) {
    for (int line = 0; line < TAILLE_BANQUISE; ++line) {
        for (int column = 0; column < TAILLE_BANQUISE; ++column) {
            printf("% 3d ", A[line][column] == INF ? -1 : A[line][column]);
        }
        cout << endl;
    }
    cout << endl;
}

template<typename T>
void print(vector<T> v) {
    for(T t : v) {
        cout << t << endl;
    }
    cout << endl;
}

v<dep> find_path(const v<v<dep>>& parent, position from, position to) {
    v<dep> deps;
    position curr = to;
    while(curr != from) {
        deps.push_back(parent[curr.ligne][curr.colonne]);
        curr = parent[curr.ligne][curr.colonne].from;
    }
    reverse(deps.begin(), deps.end());
    return deps;
}

bool is_target(position pos) {
    for(position assignement : assignments) {
        if(assignement == pos)
            return true;
    }
    return false;
}

tuple<int, position> find_nearest_alien(const v<v<int>>& dists, position pos, const v<alien_info>& active_aliens, bool must_be_free) {
    position nearest_alien;
    int min_dist = INF;
    for (alien_info alien : active_aliens) {
        if(must_be_free && is_target(alien.pos)) {
            continue;
        }
        if(dists[alien.pos.ligne][alien.pos.colonne] < min_dist) {
            min_dist = dists[alien.pos.ligne][alien.pos.colonne];
            nearest_alien = alien.pos;
        }
    }
    return {min_dist, nearest_alien};
}

void walk(int id_agent, const v<dep>& deps) {
    for(dep d : deps) {
        if(points_action_agent(id_agent) == 0) {
            break;
        }
        if(d.type == ACTION_DEPLACER) {
            deplacer(id_agent, d.dir);
        } else if (d.type == ACTION_GLISSER) {
            glisser(id_agent, d.dir);
        } else if (d.type == ACTION_POUSSER) {
            cout << "POUSSER" << endl;
            pousser(id_agent, d.dir);
            deplacer(id_agent, d.dir);
        }
    }
}

void push_ennemy(int id_agent, bool must_capture_alien) {
    position agent_pos = position_agent(moi(), id_agent);
    for(int i = 0; i < (int) dirs.size(); ++i) {
        position dest = agent_pos + dep_dirs[i];
        position behind = dest + dep_dirs[i];
        if(agent_sur_case(dest) == adversaire() && can_go(behind) && agent_sur_case(behind) == -1 &&
                (!must_capture_alien || alien_sur_case(dest))) {
            pousser(id_agent, dirs[i]);
        }
    }
}

void compute_assignments(int id_agent, const v<v<int>>& dists, const v<alien_info>& active_aliens) {
    position agent_pos = position_agent(moi(), id_agent);

    position target_alien;
    if(assignments[id_agent] == OUT) { // Si pas assigne
        // Cible la plus proche parmi celles actives et non-assignees
        int min_dist;
        tie(min_dist, target_alien) = find_nearest_alien(dists, agent_pos, active_aliens, true);
        if (min_dist == INF) {
            // Sinon cible la plus proche parmi celles actives
            tie(min_dist, target_alien) = find_nearest_alien(dists, agent_pos, active_aliens, false);
            if (min_dist == INF) { // Sinon aller sur cible en prevision TODO
                cout << "Pas de cible active" << endl;
                return;
            } else {
                cout << "Cible deja assignee" << endl;
            }
        } else {
            cout << "Cible non-assignee trouvee" << endl;
        }
        assignments[id_agent] = target_alien;
        cout << "Nouvel assignement !" << endl;
    } else { // Si assigne
        position assignement = assignments[id_agent];
        assert(targets[assignement].size() >= 1);
        int min_dist;
        tie(min_dist, target_alien) = find_nearest_alien(dists, agent_pos, active_aliens, true);
        // Si en train de capturer ou seul assigne ou pas de cible libre atteignable
        if(agent_pos == assignement  || targets[assignement].size() == 1 || min_dist == INF) {
            target_alien = assignement;
            if (target_alien == OUT)
                return;
            cout << "Garde le meme assignement" << endl;
        } else {
            cout << "Change d assignement" << endl;
        }
    }

    set_assignment(id_agent, target_alien);
    cout << "Cible : " << target_alien << endl;
}

void partie_init()
{

}

//
void jouer_tour()
{
    chrono::time_point<chrono::system_clock> start, end;
    start = chrono::system_clock::now();

    cout << "---------------------- Joueur " << moi() << " ----------------------" << endl;

    for(int id_agent = 0; id_agent < NB_AGENTS; ++id_agent) {
        position agent_pos = position_agent(moi(), id_agent);
        if(!alien_sur_case(agent_pos)) {
            remove_assignment(id_agent, agent_pos);
        }
    }

    // aliens actifs
    v<alien_info> active_aliens;
    for(alien_info alien : liste_aliens()) {
        if(alien.capture_en_cours != NB_TOURS_CAPTURE && alien.tour_invasion <= tour_actuel() &&
                tour_actuel() < alien.tour_invasion + alien.duree_invasion) {
            active_aliens.push_back(alien);
        }
    }

    cout << "Aliens cibles : " << endl;
    for (alien_info alien : active_aliens) {
        cout << alien.pos << " ";
        debug_afficher_drapeau(alien.pos, DRAPEAU_ROUGE);
    }
    cout << endl;

    for(int id_agent = 0; id_agent < NB_AGENTS; ++id_agent) {
        position agent_pos = position_agent(moi(), id_agent);
        cout << "================ Agent : " << agent_pos << " ================" << endl;

        // Tableau Dijkstra
        v<v<int>> dists;
        v<v<dep>> parent;
        tie(dists, parent) = min_move(agent_pos);
#ifdef DEBUG
        print_vvi(dists);
#endif
        compute_assignments(id_agent, dists, active_aliens);
        position assignment = assignments[id_agent];

        if(assignment != OUT) {
            // Trouver chemin
            v<dep> deps = find_path(parent, agent_pos, assignment);
#ifdef DEBUG
            cout << "Path :" << endl;
            for(dep d : deps) {
                cout << d << endl;
            }
            cout << endl;
#endif
            // Parcourir chemin
            walk(id_agent, deps);

            agent_pos = position_agent(moi(), id_agent); // L'agent a bouge
        }

        // Pousser si sur alien
        push_ennemy(id_agent, true);
        // Sinon peut pousser aussi
        push_ennemy(id_agent, false);

        end = chrono::system_clock::now();
        chrono::duration<double> elapsed_seconds = end - start;
        cout << "================ Time : " << elapsed_seconds.count() << " ================" << endl;
    }

    for (alien_info alien : active_aliens) {
        debug_afficher_drapeau(alien.pos, AUCUN_DRAPEAU);
    }
}

void partie_fin()
{

}






















