#include "prologin.hh"
#include "debug.hh"
#include "structures.hh"
#include "coefs.hh"
#include <vector>
#include <queue>
#include <iostream>
#include <algorithm>
#include <string>
#include <cstdio>
#include <deque>
using namespace std;


/**
 * Mon IA est une heuristique qui se base principalement sur deux manières d'attribuer des scores à des *cases* :
 * - le Dijkstra, utilisé pour déterminer les chemins optimaux à emprunter pour atteindre chaque situation,
 * - une fonction de score par case qui est principalement basée sur la valeur d'un éventuel minerai présent dans la case,
 *   atténuée par la distance de la case et la résistance du minerai entre autres.
 * La plupart des paramètres rentrant en jeu dans mon IA peuvent être réglés dans le fichier "coefs.hh".
 * Les nains sont relativement indépendants, malgré le fait qu'ils essaient de ne pas trop se regrouper.
 * 
 * Le déroulement d'un tour est décrit plus précisément au-dessus du code de la fonction "jouer_tour", la
 * dernière de ce fichier.
 * 
 * Idées non abouties :
 * - Prise en compte de la présence d'ennemis à proximité (baisse les scores)
 * - Tirage de corde (jamais envisagé)
 * - Dépôt de corde moins au hasard (jamais envisagé)
 * - Coopération plus poussée entre nains (par exemple pour tuer des ennemis ou miner un minerai long à miner) (jamais envisagé)
 * - Retour plus fin à la taverne à la fin de la partie (jamais envisagé)
 */


// Toute ligne préfixée par "if(afficher)" peut afficher un débug en HTML.
#ifdef LOCAL
bool afficher = true;
#else
bool afficher = false;
#endif

void partie_init() {}
void partie_fin() {}

/**
 * Fonctions utilitaires diverses.
 */

// Case suivante dans une direction.
position caseSuiv(position pos, direction dir) {
	switch (dir) {
	case HAUT:
		return {pos.ligne-1, pos.colonne};
	case DROITE:
		return {pos.ligne, pos.colonne+1};
	case BAS:
		return {pos.ligne+1, pos.colonne};
	case GAUCHE:
		return {pos.ligne, pos.colonne-1};
	default:
		return {-1,-1};
	}
}

// Distance de Manhattan.
int distanceMan(position p1, position p2) {
	return abs(p1.ligne-p2.ligne) + abs(p1.colonne-p2.colonne);
}

// Vérification de limites de carte.
bool valide(position pos) {
	return pos.ligne >= 0 && pos.ligne < TAILLE_MINE && pos.colonne >= 0 && pos.colonne < TAILLE_MINE;
}

// Vérifie si un nain à la position donnée peut tomber.
bool estSol(position pos) {
	if(pos.ligne == TAILLE_MINE-1)
		return true;
	position dessous = caseSuiv(pos, BAS);
	return type_case(dessous) != LIBRE || nain_sur_case(dessous) == adversaire();
}

// Nombre de cases entre la position et le sol.
// Pour une case au sol au sens de la fonction précédente, renvoie 0.
int descente(position pos) {
	int ret = 0;
	while(!estSol(pos)) {
		pos.ligne++;
		ret++;
	}
	return ret;
}

// Coups de pioche nécessaires pour détruire une case (inclut un éventuel nain).
int coupsPioche(position pos) {
	/*int maxVieNain = 0;
	for(int iNain = 0; iNain < NB_NAINS; iNain++) {
		nain info = info_nain(adversaire(), iNain);
		if(info.pos == pos)
			maxVieNain = max(maxVieNain, info.vie);
	}
	if(maxVieNain > 0)
		return (maxVieNain - 1) / DEGAT_PIOCHE + 1;*/ // Rollback : diminuait les scores
	if(type_case(pos) == LIBRE)
		return 0;
	else if(type_case(pos) == OBSIDIENNE)
		return INFINI;
	minerai mine = info_minerai(pos);
	return max(1, mine.resistance);
}




/**
 * Fonctions d'attribution de scores
 */

// Attribue un malus relatif à la proximité d'une case avec un ennemi (protection des nains).
// Formule : somme de (écart à la prise en compte * malus fixe).
// Ignoré dans la version actuelle.
int malusProximite(position pos) {
	int total = 0;
	for(int iNain = 0; iNain < NB_NAINS; iNain++) {
		nain info = info_nain(adversaire(), iNain);
		int d = distanceMan(info.pos, pos);
		if(d <= DIST_PROX_NENNEMI)
			total += (DIST_PROX_NENNEMI-d)*MALUS_PROX_NENNEMI;
	}
	return total;
}

/**
 * Coût d'un déplacement. Si limiter = true, interdit tout minage (dépôt de corde).
 * Composantes (mis à part les cas non désirés) :
 * - Si la case est déjà occupée par un nain, malus en fonction de l'équipe du nain.
 * - Point de mouvement. Si on est accroché, multiplier par 2.
 * - Points d'action (minage d'un nain ou d'une case).
 */
int coutDepl(position pos, bool accroche, direction dir, bool limiter = false) {
	position suiv = caseSuiv(pos, dir);
	position pos_tadv = position_taverne(adversaire());
	// Situations que l'on ne veut absolument pas atteindre.
	if(	!valide(suiv) || // Hors de la carte
		(!accroche && descente(suiv) > CHUTE_SAFE) || // Mort (ou blessure)
		(limiter && type_case(suiv) != LIBRE) || // Si on ne veut pas de minage
		type_case(suiv) == OBSIDIENNE || // Impossible
		posEgales(suiv, pos_tadv) ) // Taverne adverse (!)
		return INFINI;
	int malus = 0;
	if(nain_sur_case(suiv) == moi())
		malus = MALUS_TR_NALLIE;
	if(nain_sur_case(suiv) == adversaire())
		malus = MALUS_TR_NENNEMI;
	if(type_case(suiv) == LIBRE) {
		if(accroche) {
			if(corde_sur_case(pos) && corde_sur_case(suiv))
				//return malus + COUT_MOUV + coupsPioche(suiv)*COUT_MINER*COUT_ACTION; // Rollback
				return COUT_MOUV;
			else
				//return malus + 2*COUT_MOUV + coupsPioche(suiv)*COUT_MINER*COUT_ACTION; // Rollback
				return 2*COUT_MOUV;
		}
		else
			return COUT_MOUV;
	}
	return malus + max(0, coupsPioche(suiv)*COUT_MINER*COUT_ACTION + (!accroche && dir == BAS ? 0 : COUT_MOUV));
}

// Crée la liste des situations suivantes à partir d'une situation donnée.
// Si limite = true, interdit tout minage.
vector<Situation> sitsSuivantes(Situation sit, bool limiter = false) {
	position pos_tadv = position_taverne(adversaire());
	vector<Situation> ret;
	// 4 directions
	for(direction dir : {HAUT, BAS, GAUCHE, DROITE}) {
		// Il faut s'accrocher pour monter
		if(!sit.accroche && dir == HAUT)
			continue;
		int cost = coutDepl(sit.pos, sit.accroche, dir, limiter);
		if(cost != INFINI) {
			position suiv = caseSuiv(sit.pos, dir);
			if(!sit.accroche)
				suiv.ligne += descente(suiv);
			if(posEgales(suiv, pos_tadv))
				continue;
			ret.push_back({suiv, sit.accroche, sit.cost+cost, {ACTION_DEPLACER, -1, dir, dir}, coupsPioche(caseSuiv(sit.pos, dir)), sit.pos, sit.accroche, sit.pa+coupsPioche(caseSuiv(sit.pos, dir))*COUT_MINER});
		}
	}
	// Descente, si on ne tombe pas sur quelque chose de mal
	int desc = descente(sit.pos);
	if(sit.accroche && desc <= CHUTE_SAFE && (sit.pos.ligne+desc != pos_tadv.ligne || sit.pos.colonne != pos_tadv.colonne)) {
		position suiv = {sit.pos.ligne+desc, sit.pos.colonne};
		ret.push_back({suiv, false, sit.cost+COUT_LACHER*COUT_ACTION, {ACTION_LACHER, -1, HAUT, HAUT}, 0, sit.pos, sit.accroche, sit.pa});
	}
	// Accrochage
	if(!sit.accroche)
		ret.push_back({sit.pos, true, sit.cost+COUT_AGRIPPER*COUT_ACTION, {ACTION_AGRIPPER, -1, HAUT, HAUT}, 0, sit.pos, sit.accroche, sit.pa});
	return ret;
}

// Dijkstra : détermination d'un chemin vers chaque case atteignable
vector<vector<vector<Situation> > > coutsDepl(int idnain, bool limiter = false) {
	priority_queue<Situation> sits;
	nain init = info_nain(moi(), idnain);
	Situation sitInit = {init.pos, init.accroche, 0, {ACTION_LACHER, -1, HAUT, HAUT}, 0, {-1, -1}, false, 0};
	vector<vector<vector<Situation> > > ret(TAILLE_MINE, vector<vector<Situation>> (TAILLE_MINE, vector<Situation> (2, {init.pos, false, NON_VISITE, {ACTION_LACHER, -1, HAUT, HAUT}, 0, {-1, -1}, false, 0})));
	sits.push(sitInit);
	while(!sits.empty()) {
		Situation sit = sits.top();
		sits.pop();
		if(ret[sit.pos.ligne][sit.pos.colonne][sit.accroche].cost < NON_VISITE)
			continue;
		ret[sit.pos.ligne][sit.pos.colonne][sit.accroche] = sit;
		vector<Situation> suivs = sitsSuivantes(sit, limiter);
		for(Situation suiv : suivs)
			sits.push(suiv);
	}
	return ret;
}

// Somme des butins des nains adversaires en une certaine position
int sommeNainAdverse(position pos) {
	int s = 0;
	for(int iNain = 0; iNain < NB_NAINS; iNain++) {
		nain ret = info_nain(adversaire(), iNain);
		if(ret.pos.ligne == pos.ligne && ret.pos.colonne == pos.colonne)
			s += ret.butin;
	}
	return s;
}

/**
 * La fonction qui attribue un score par situation.
 * Composantes :
 * - Si on a du butin et peu de vie, retour immédiat à la taverne
 * - Si on est à la fin du jeu, retour immédiat à la taverne
 * - Si on a assez de butin, retour immédiat à la taverne
 * - Taverne adverse : jamais de la vie
 * - Nain allié : malus (pour éviter qu'un nain adversaire ne tue tout le monde)
 * - Proximité à l'ennemi : malus (a priori, le coefficient est trop petit pour que l'impact soit réel)
 * - Prise en compte de la présence de minerais ennemis : valeur / résistance / distance (cf code)
 * - Prise en compte de la présence d'un nain ennemi attaquable à portée.
 * - Si on est vulnérable ou convoité, on évite une case adjacente à un nain !
 * - Bonus ridicule pour la taverne (s'il n'y a rien à faire)
 */
long long scoreSit(vector<vector<vector<Situation> > > & sits, int inain, position pos, bool accroche) {
	long long total = 0;
	long long dist = sits[pos.ligne][pos.colonne][accroche].cost;
	int pa = sits[pos.ligne][pos.colonne][accroche].pa;
	nain info = info_nain(moi(), inain);
	position pos_tav = position_taverne(moi());
	position pos_tavadv = position_taverne(adversaire());
	// Taverne adversaire
	if(pos == pos_tavadv)
		return SCORE_TADV;
	// Allié sur la case
	if(nain_sur_case(pos) == moi() && !posEgales(pos, info.pos))
		total -= MALUS_SIT_NALLIE;
	// Malus de proximité d'un ennemi (très peu d'influence)
	if(dist <= DIST_PROX_NENNEMI)
		total -= malusProximite(pos);
	// Cas spéciaux de retour immédiat à la taverne
	if((info.butin > 0 && info.vie <= VIE_RETOUR) || info.butin >= BUTIN_RETOUR || tour_actuel() >= TOUR_RUSH_TAV) {
		if(pos_tav.ligne == pos.ligne && pos_tav.colonne == pos.colonne)
			return INFINI;
		else
			return -INFINI;
	}
	// Exploration des cases immédiatement adjacentes : minerais, nains
	for(direction dir : {HAUT, GAUCHE, BAS, DROITE}) {
		position suiv = caseSuiv(pos, dir);
		if(!valide(suiv))
			continue;
		if(type_case(suiv) == GRANITE && (accroche || dir != BAS || descente(suiv) < CHUTE_SAFE-1)) {
			minerai mine = info_minerai(suiv);
			if(mine.resistance >= 0)
				total += min(BUTIN_MAX-info.butin, mine.rendement) * COEF_RENDEMENT / max(1, (mine.resistance/COEF_DIV_RESISTANCE)) / max(1ll, COEF_COUT*dist); 
		}
		// Attaque (ou fuite) d'un nain ennemi
		if(dist <= DIST_ATTAQUE_ENNEMI && nain_sur_case(suiv) == adversaire() && pa == 0 && distanceMan(suiv, pos_tavadv) > MIN_DISTANCE_TADV_ATQ) {
			debug_afficher_drapeau(suiv, DRAPEAU_ROUGE);
			if(info.vie < VIE_NAIN || info.butin > 0)
				total = NON_ATTAQUE;
			else
				total += (CONSTANTE_ADV+sommeNainAdverse(suiv))*COEF_ATTAQUE;
		}
	}
	// Sauf cas exceptionnels repris au-dessus, la taverne a un score ridicule, mais positif
	if(pos == pos_tav)
		total = SCORE_TAV;
	return total;
}





/**
 * Fonctions d'actions (celles qui font faire des choses aux nains)
 */

// Crée une liste d'instructions à exécuter pour atteindre une certaine situation.
// Cette liste sera dans la plupart des cas trop longue pour être exécutée en un tour.
deque<action_hist> listeInstr(vector<vector<vector<Situation> > > &mat, int iNain, position arrivee, bool accroche) {
	if(afficher)
		cout << "Vers " << arrivee.ligne << " " << arrivee.colonne << " " << (accroche ? "acc" : "noacc") << "<br>\n" << flush;
	nain info = info_nain(moi(), iNain);
	deque<action_hist> ret;
	Situation actuel = mat[arrivee.ligne][arrivee.colonne][accroche];
	while(actuel.pos.ligne != info.pos.ligne || actuel.pos.colonne != info.pos.colonne || actuel.accroche != info.accroche) {
		if(actuel.accroche || actuel.action.atype != ACTION_DEPLACER || actuel.action.dir != BAS)
			ret.push_front(actuel.action);
		for(int i = 0; i < actuel.nMine; i++)
			ret.push_front({ACTION_MINER, iNain, actuel.action.dir, actuel.action.dir});
		if(afficher)
			cout << "Etape : " << actuel.posPrec.ligne << " " << actuel.posPrec.colonne << " " << (actuel.accPrec ? "acc" : "noacc") << "<br>\n" << flush;
		actuel = mat[actuel.posPrec.ligne][actuel.posPrec.colonne][actuel.accPrec];
	}
	return ret;
}

// Le 1er nain déposera une corde la première fois qu'il pourra, si la corde résultante
// dépasse une certaine longueur.
bool cordePosee = false;
bool poserCorde(int iNain = 0) {
	vector<vector<vector<Situation> > > atteignables = coutsDepl(iNain, true);
	// Recherche d'une position de corde optimale
	int lgMax = 0;
	position corres = {0, 0};
	direction dirMax = HAUT;
	bool accrocheMax = false;
	for(int lig = 0; lig < TAILLE_MINE; lig++)
		for(int col = 0; col < TAILLE_MINE; col++)
			for(bool accroche : {false, true})
				if(atteignables[lig][col][accroche].cost != NON_VISITE && atteignables[lig][col][accroche].cost <= NB_POINTS_DEPLACEMENT)
					for(direction dir : {HAUT, GAUCHE, BAS, DROITE}) {
						position suiv = caseSuiv({lig, col}, dir);
						if(!valide(suiv) || type_case(suiv) != LIBRE)
							continue;
						int desc = descente(suiv)+1;
						if(desc >= lgMax) {
							lgMax = desc;
							corres = {lig, col};
							dirMax = dir;
							accrocheMax = accroche;
						}
					}
	if(lgMax <= LG_MIN_CORDE)
		return false;
	// Placement de la corde
	if(afficher)
		cout << "<p>On peut placer une corde ! Détermination instructions...<br>\n" << flush;
	deque<action_hist> actions = listeInstr(atteignables, iNain, corres, accrocheMax);
	if(afficher)
		cout << "</p>\n" << flush;
	if(afficher) {
		cout << "<p>Actions d&eacute;cid&eacute;es : ";
		for(action_hist act : actions)
			cout << act_str(act) << " ";
		cout << "</p>\n" << flush;
	}
	for(action_hist act : actions) {
		erreur err;
		if(act.atype == ACTION_DEPLACER)
			err = deplacer(iNain, act.dir);
		else if(act.atype == ACTION_MINER || act.atype == ACTION_POSER_CORDE || act.atype == ACTION_TIRER) {
			cout << "<p style=\"color: red\">Action interdite !</p>\n" << flush;
			break;
		}
		else if(act.atype == ACTION_AGRIPPER)
			err = agripper(iNain);
		else if(act.atype == ACTION_LACHER)
			err = lacher(iNain);
		if(err != OK)
			break;
	}
	poser_corde(iNain, dirMax);
	return true;
}



/**
 * Déroulement d'un tour :
 * Le traitement est effectué nain par nain.
 * Pour chaque nain, tant qu'il peut faire quelque chose, on lui fait faire quelque chose (dans la limite de trois essais),
 * dans l'ordre de la liste ci-dessous :
 * - Lancement d'un Dijkstra à partir de la position du nain
 * - Détermination de la case à score maximal
 * - Exécution des instructions requises (dans la limite du faisable)
 * - Auto-défense : attaque des nains adjacents
 * - Minage au "hasard" (je n'ai pas eu le temps d'optimiser et de tester l'ordre)
 * Le dernier nain dépose une corde losque c'est possible (ceci n'impactera pas le reste du tour).
 */
void jouer_tour() {
	if(afficher) {
		init_debug();
		cout << "<h1>Tour n° " << tour_actuel() << "\t</h1>\n" << flush;
		afficherMap();
	}
	for(int iNain = 0; iNain < NB_NAINS; iNain++) {
		nain infoprec = info_nain(moi(), iNain);
		nain infoact = info_nain(moi(), iNain);
		int essai = 0;
		do {
			infoprec = infoact;
			if(afficher)
				cout << "<h2>-&gt; Traitement nain " << iNain << "</h2>\n" << flush;
			position posAct = info_nain(moi(), iNain).pos;
			if(posAct.ligne == -1) {
				if(afficher)
					cout << "<p>Nain mort, on passe</p>";
				continue;
			}
			// Dépôt de corde du 1er nain
			if(!cordePosee && iNain == 0) {
				if(afficher)
					cout << "<p>Essai de dépôt de corde...</p>\n" << flush;
				if(poserCorde()) {
					if(afficher)
						cout << "<p>Réussi !</p>\n" << flush;
					cordePosee = true;
					continue;
				}
			}
			// Dijkstra
			if(afficher)
				cout << "<p>Lancement Dijkstra...<br>\n" << flush;
			vector<vector<vector<Situation> > > mat = coutsDepl(iNain);
			// Score optimal
			if(afficher)
				cout << "</p><p>Détermination situation optimale... ";
			vector<vector<vector<long long> > > scores(TAILLE_MINE, vector<vector<long long> > (TAILLE_MINE, vector<long long> (2)));
			long long maxi = -INFINI, mini = INFINI;
			position posCorres = {0, 0};
			bool accCorres = false;
			bool trouve = false;
			for(int lig = 0; lig < TAILLE_MINE; lig++)
				for(int col = 0; col < TAILLE_MINE; col++) {
					for(bool accroche : {false, true}) {
						long long sc = scoreSit(mat, iNain, {lig, col}, accroche);
						scores[lig][col][accroche] = sc;
						if(sc > maxi && mat[lig][col][accroche].cost != NON_VISITE) {
							trouve = true;
							maxi = sc;
							posCorres = {lig, col};
							accCorres = accroche;
						}
						if(sc < mini && sc != SCORE_TADV)
							mini = sc;
					}
				}
			if(!trouve) {
				if(afficher)
					cout << "Aucune situation accessible (!!).</p>\n" << flush;
				continue;
			}
			if(afficher) {
				cout << "Se situe en (" << posCorres.ligne << "," << posCorres.colonne << "," << (accCorres ? "acc" : "nonacc") << "), vaut " << maxi << "\n" << flush;
				afficherGrille(mat, scores, iNain, maxi, mini);
				cout << "<p>Détermination instructions...<br>\n" << flush;
			}
			debug_afficher_drapeau(posCorres, DRAPEAU_BLEU);
			// On effectue les déplacements/minages nécessaires, dans la mesure du possible
			deque<action_hist> actions = listeInstr(mat, iNain, posCorres, accCorres);
			if(afficher) {
				cout << "</p>\n" << flush;
				cout << "<p>Actions d&eacute;cid&eacute;es : ";
				for(action_hist act : actions)
					cout << act_str(act) << " ";
				cout << "</p>\n" << flush;
			}
			for(action_hist act : actions) {
				erreur err;
				if(act.atype == ACTION_DEPLACER)
					err = deplacer(iNain, act.dir);
				else if(act.atype == ACTION_MINER)
					err = miner(iNain, act.dir);
				else if(act.atype == ACTION_AGRIPPER)
					err = agripper(iNain);
				else if(act.atype == ACTION_LACHER)
					err = lacher(iNain);
				else if(act.atype == ACTION_POSER_CORDE)
					err = poser_corde(iNain, act.dir);
				else if(act.atype == ACTION_TIRER)
					err = tirer(iNain, act.dir, act.sens);
				if(err != OK)
					break;
			}
			nain nn = info_nain(moi(), iNain);
			posAct = nn.pos;
			debug_afficher_drapeau(posAct, DRAPEAU_VERT);
			// Priorité à l'auto-défense/à l'attaque
			for(direction dir : {HAUT, BAS, GAUCHE, DROITE}) {
				position posMine = caseSuiv(posAct, dir);
				if(!valide(posMine))
					continue;
				erreur err = OK;
				while(err == OK && nain_sur_case(posMine) == adversaire())
					err = miner(iNain, dir);
			}
			// Minage
			for(direction dir : {HAUT, BAS, GAUCHE, DROITE}) {
				position posMine = caseSuiv(posAct, dir);
				if(!valide(posMine))
					continue;
				minerai mine = info_minerai(posMine);
				erreur err = OK;
				if(!nn.accroche && dir == BAS && descente(posMine) >= CHUTE_SAFE - 1)
					continue;
				//debug_afficher_drapeau(posMine, moi() == 1 ? DRAPEAU_VERT : DRAPEAU_ROUGE);
				while(err == OK && mine.rendement >= 1) {
					err = miner(iNain, dir);
					mine = info_minerai(posMine);
				}
			}
			// Dépôt de corde pour le dernier nain
			nn = info_nain(moi(), iNain);
			if(iNain == NB_NAINS-1 && nn.pa == NB_POINTS_ACTION) {
				int maxi = 0;
				direction corres = HAUT;
				for(direction dir : {HAUT, GAUCHE, BAS, DROITE}) {
					position suiv = caseSuiv(nn.pos, dir);
					if(valide(suiv) && type_case(suiv) == LIBRE && descente(suiv) >= maxi) {
						maxi = descente(suiv);
						corres = dir;
					}
				}
				poser_corde(iNain, corres);
			}
			if(afficher)
				finish_debug();
			infoact = info_nain(moi(), iNain);
			essai++;
		} while ((infoact.pm != infoprec.pm || infoact.pa != infoprec.pa) && essai < 3);
	}
}


