import enum
from math import ceil
from typing import List, Dict, Tuple

Pos = Tuple[int, int]
Map = Dict[Pos, "Case"]
INF = float('inf')

import api
from utils import *

OK = api.erreur.OK

class CaseType(enum.IntFlag):
    VIDE =          0b0001
    UNBREAKABLE =   0b0100
    OBSTACLE =      0b1000

    ROPE =          0b0010
    TAVERNE_MOI =   0b0011
    TAVERNE_ENEMI = 0b0101
    GRANITE =       0b1001
    MINERAI =       0b1010
    OBSIDIENNE =    0b1100


class Nain:
    id: int
    pos: Pos
    vie: int
    pa: int
    pm: int
    accroche: bool
    butin: int

    def __init__(self, id, pos, vie, pa, pm, accroche, butin):
        self.id = id
        self.pos = pos
        self.vie = vie
        self.pa = pa
        self.pm = pm
        self.accroche = accroche
        self.butin = butin

    def __str__(self):
        return f"Nain({self.id}, pos={self.pos}, butin={self.butin}, PM={self.pm})"

    @classmethod
    def load(cls, id, player):
        n = api.info_nain(player, id)
        nain = cls(id, n.pos, n.vie, n.pa, n.pm, n.accroche, n.butin)
        # debug(id, n)
        return nain

    def move(self, dir, for_real=True):
        """Return whether the move failed"""
        dir = DIR_TO_API[dir]
        cout = api.cout_de_deplacement(self.id, dir)
        if for_real:
            error = api.deplacer(self.id, dir)
        else:
            error = api.erreur.OK  # TODO: checks

        if error == api.erreur.OK:
            self.pm -= cout
            self.pos = add(self.pos, DIR_FROM_API[dir])

        return error != api.erreur.OK

    def grip(self, for_real=True):
        if for_real:
            api.agripper(self.id)
        return True

    def lacher(self, for_real=True):
        if for_real:
            error = api.lacher(self.id)
        else:
            error = OK

        if error == OK:
            n = api.info_nain(MOI, self.id)
            self.__init__(self.id, n.pos, n.vie, n.pa, n.pm, n.accroche, n.butin)

        return self.vie <= 0

    def rope(self, dir, for_real=True):
        if for_real:
            error = api.poser_corde(self.id, DIR_TO_API[dir])
        else:
            error = api.erreur.OK

        return error != api.erreur.OK


class Minerai:
    resi: int
    butin: int
    pos: Pos

    def __init__(self, res, butin, pos):
        self.resi = res
        self.butin = min(butin, api.BUTIN_MAX)
        self.pos = pos

    @classmethod
    def load(cls, pos):
        m = api.info_minerai(pos)
        return cls(m.resistance, m.rendement, pos)

    def mine(self):
        self.resi -= 1
        if self.resi <= 0:
            return self.butin


class Case:
    pos: Pos
    type: CaseType
    minerai: Minerai

    def __init__(self, pos, type, minerai=None):
        self.pos = pos
        self.type = type
        self.minerai = minerai

    def __str__(self):
        return f"Case({repr(self.type)}, {self.pos})"

    def mine(self, game):
        if self.type == CaseType.GRANITE:
            self.type = CaseType.VIDE
            game.expand_rope(add(self.pos, UP))
        elif self.type == CaseType.MINERAI:
            self.minerai.resi -= 1
            if self.minerai.resi <= 0:
                self.type = CaseType.VIDE
                self.minerai = None
                game.expand_rope(add(self.pos, UP))
                return game.minerais.pop(self.pos).butin
        return 0

    @property
    def resi(self):
        if self.minerai:
            return self.minerai.resi
        if self.type == CaseType.GRANITE:
            return 1
        if self.type & CaseType.UNBREAKABLE:
            return INF
        if self.type == CaseType.TAVERNE_ENEMI:
            return 101
        return 0

    @property
    def char(self):
        t = self.type
        if t == CaseType.OBSIDIENNE:
            return '#'
        elif t & CaseType.OBSTACLE:
            return '@'
        elif t == CaseType.ROPE:
            return '|'
        else:
            return ' '


class Game:
    nains: List[Nain]
    enemi: List[Nain]
    minerais: Dict[Pos, Minerai]
    map: Map
    my_taverne: Pos
    enemi_taverne: Pos

    def __init__(self, nains, enemi, minerais, my_taverne, enemi_taverne, map):
        self.nains = nains
        self.enemi = enemi
        self.minerais = minerais
        self.my_taverne = my_taverne
        self.enemi_taverne = enemi_taverne
        self.map = map


        # sentinelles
        for l in range(-1, api.TAILLE_MINE + 1):
            for c in range(-1, api.TAILLE_MINE + 1):
                self.map.setdefault((l, c), Case((l, c), CaseType.OBSIDIENNE))

        # self.print_map()

    @classmethod
    def load_map(cls, minerai: Dict[Pos, Minerai]):
        carte = {}
        for l in range(api.TAILLE_MINE):
            for c in range(api.TAILLE_MINE):
                carte[l, c] = cls.load_case(l, c)

        carte[api.position_taverne(api.moi())].type = CaseType.TAVERNE_MOI
        carte[api.position_taverne(api.adversaire())].type = CaseType.TAVERNE_ENEMI

        for pos, m in minerai.items():
            carte[pos].type = CaseType.MINERAI
            carte[pos].minerai = m

        for pos in api.liste_cordes():
            carte[pos].type = CaseType.ROPE
            
        return carte

    @classmethod
    def load(cls):
        minerai = {pos: Minerai.load(pos) for pos in api.liste_minerais()}
        return cls([Nain.load(i, api.moi()) for i in range(api.NB_NAINS)],
                   [Nain.load(i, api.adversaire()) for i in range(api.NB_NAINS)],
                   minerai,
                   api.position_taverne(api.moi()), api.position_taverne(api.adversaire()),
                   cls.load_map(minerai))

    @classmethod
    def load_case(cls, l, c):
        if l < 0 or l >= api.TAILLE_MINE or c < 0 or c >= api.TAILLE_MINE:
            return Case((l, c), CaseType.OBSIDIENNE)

        info: api.case_type = api.type_case((l, c))

        type = {
            api.case_type.OBSIDIENNE: CaseType.OBSIDIENNE,
            api.case_type.GRANITE: CaseType.GRANITE,
            api.case_type.LIBRE: CaseType.VIDE
        }[info]

        return Case((l, c), type)

    def case(self, pos, dir=(0, 0)):
        return self.map[pos[0] + dir[0], pos[1] + dir[1]]

    def mine(self, nain: Nain, dir, for_real=True):

        if for_real:
            nain.grip()
            error = api.miner(nain.id, DIR_TO_API[dir])
        else:
            error = api.erreur.OK  # TODO: checks

        if error == api.erreur.OK:
            target = add(nain.pos, dir)

            # mine ore
            nain.pa -= api.COUT_MINER
            case = self.map[target]
            nain.butin += case.mine(self)
            # mine dwarves
            for en in self.enemi[:]:
                if en.pos == target:
                    en.vie -= api.DEGAT_PIOCHE
                    if en.vie < 0:
                        self.enemi.remove(en)

        return error != api.erreur.OK

    def move_cost(self, pos, dir, case_type_override=None):
        """
        Cost of move and necesssity of grip
        :return int, bool: Cost and whether we should grip
        """

        dest = self.case(pos, dir)
        if case_type_override is None:
            t = dest.type
        else:
            t = case_type_override

        if case_type_override is None and dest.pos in {e.pos for e in self.enemi}:
            return -1, True
        if t == CaseType.OBSIDIENNE:
            return INF, True
        if t & CaseType.OBSTACLE:
            return -1, True
        if t == CaseType.ROPE:
            return 1, True
        if dir in (LEFT, RIGHT):
            bot = self.case(pos, DOWN)
            next = self.case(bot.pos, dir)
            if bot.type & CaseType.OBSTACLE and next.type & CaseType.OBSTACLE:
                return 1, False

        return 2, True

    def print_map(self):
        for l in range(api.TAILLE_MINE):
            for c in range(api.TAILLE_MINE):

                debug(self.map[l, c].char, end='')
            debug()

    def pp_dist(self, costs):
        for l in range(-1, 31):
            print('{:<3}'.format(l), end='')
            for c in range(-1, 31):
                print(self.map[l, c].char, end='')
            print('  ', end='')
            print('{:<3}'.format(l), end='')
            for c in range(-1, 31):
                d = costs[l, c]
                bg = d[0] % 8
                bold = 37 if d[1] else 30
                print(f'\033[4{bg};{bold}m', chr(ord('0') + d[0]), end='\033[m', sep='')
            print()

    def rope(self, nain, dir, for_real=True):
        if for_real:
            error = api.poser_corde(nain.id, DIR_TO_API[dir])
        else:
            error = OK
            
        if error == OK:
            for n in self.nains:
                n.pa = 0
            target = add(nain.pos, dir)
            self.map[target].type = CaseType.ROPE
            self.expand_rope(target)

        return error != OK

    def expand_rope(self, target):
        if self.map[target].type != CaseType.ROPE:
            return

        l, c = target
        l += 1
        case = self.map[l, c]
        while case.type == CaseType.VIDE:
            case.type = CaseType.ROPE
            l += 1
            case = self.map[l, c]

    @classmethod
    def load_file(cls, file):

        carte = {}  # type: Dict[Pos, Case]
        with open(file) as f:
            for l in range(31):
                line = f.readline()
                for c in range(31):
                    t = {'.': CaseType.VIDE,
                         'X': CaseType.GRANITE,
                         '#': CaseType.OBSIDIENNE}[line[c]]

                    carte[l, c] = Case((l, c), t)

            # Taverne
            ma_taverne = tuple(map(int, f.readline().split()))
            en_taverne = tuple(map(int, f.readline().split()))
            carte[ma_taverne].type = CaseType.TAVERNE_MOI
            carte[en_taverne].type = CaseType.TAVERNE_ENEMI

            # Minerai
            minerais = {}  # type: Dict[Pos, Minerai]
            for _ in range(int(f.readline())):
                l, c, res, but = map(int, f.readline().split())
                minerais[l, c] = Minerai(res, but, (l, c))

            for pos, m in minerais.items():
                carte[pos].type = CaseType.MINERAI
                carte[pos].minerai = m

            # Cordes
            cordes = []
            for _ in range(int(f.readline())):
                pos = tuple(map(int, f.readline().split()))
                carte[pos].type = CaseType.ROPE
                cordes.append(pos)

            game = cls([], [], minerais, ma_taverne, en_taverne, carte)
            for pos in cordes:
                game.expand_rope(pos)

        return game

    def voisins(self, pos, type=None):
        for dir in DIRECTIONS:
            case = self.case(pos, dir)
            if type is None or case.type == type:
                yield case.pos, dir

    def turns_to_break(self, pos):
        return self.map[pos].resi or max((ceil(e.vie / 3) for e in self.enemi if e.pos == pos), default=0)
