#![allow(unused)]
use ffi::{self, ToAPI, ToFFI};
use std::os::raw::{c_float, c_double, c_int, c_void};
use state::*;

/// Taille de la banquise (longueur et largeur).
pub const TAILLE_BANQUISE: c_int = 25;

/// Nombre de cases totales
pub const NB_CASES: c_int = TAILLE_BANQUISE * TAILLE_BANQUISE;

/// Nombre de tours à jouer avant la fin de la partie.
pub const NB_TOURS: c_int = 100;

/// Nombre de points d'action par tour par agent.
pub const NB_POINTS_ACTION: c_int = 8;

/// Nombre de point d'action que coûte un déplacement.
pub const COUT_DEPLACEMENT: c_int = 1;

/// Nombre de points d'action que coûte une glissade.
pub const COUT_GLISSADE: c_int = 3;

/// Nombre de points d'action nécessaires pour pousser un agent.
pub const COUT_POUSSER: c_int = 5;

/// Nombre d'agents par joueur.
pub const NB_AGENTS: usize = 4;

/// Nombre de tours nécessaires pour capturer un alien.
pub const NB_TOURS_CAPTURE: c_int = 3;

pub const PLAYERS: &[c_int] = &[1, 2];

pub const AGENTS: &[c_int] = &[0, 1, 2, 3];

/// Types de cases
#[repr(C)]
#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq, PartialOrd, Ord)]
pub enum CaseType {
  /// Case libre
  Libre,
  /// Mur
  Mur,
  /// Erreur
  Erreur,
}

/// Points cardinaux
#[repr(C)]
#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq, PartialOrd, Ord)]
pub enum Direction {
  /// Direction : nord
  Nord,
  /// Direction : est
  Est,
  /// Direction : sud
  Sud,
  /// Direction : ouest
  Ouest,
}

/// Erreurs possibles
#[repr(C)]
#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq, PartialOrd, Ord)]
pub enum Erreur {
  /// L'action s'est effectuée avec succès.
  Ok,
  /// Votre agent ne possède pas assez de points d'action pour réaliser cette action.
  PaInsuffisants,
  /// La position spécifiée n'est pas sur la banquise.
  PositionInvalide,
  /// La position spécifiée est un mur.
  ObstacleMur,
  /// La position spécifiée est un agent.
  ObstacleAgent,
  /// Ce déplacement fait sortir un agent des limites de la banquise.
  DeplacementHorsLimites,
  /// La direction spécifiée n'existe pas.
  DirectionInvalide,
  /// L'agent spécifié n'existe pas.
  IdAgentInvalide,
  /// Aucun agent à pousser dans la direction indiquée.
  RienAPousser,
  /// Le drapeau spécifié n'existe pas.
  DrapeauInvalide,
}

/// Types d'actions
#[repr(C)]
#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq, PartialOrd, Ord)]
pub enum ActionType {
  /// Action ``deplacer``
  ActionDeplacer,
  /// Action ``glisser``
  ActionGlisser,
  /// Action ``pousser``
  ActionPousser,
}

/// Types de drapeaux de débug
#[repr(C)]
#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq, PartialOrd, Ord)]
pub enum DebugDrapeau {
  /// Aucun drapeau, enlève le drapeau présent
  AucunDrapeau,
  /// Drapeau bleu
  DrapeauBleu,
  /// Drapeau vert
  DrapeauVert,
  /// Drapeau rouge
  DrapeauRouge,
}

/// Position sur la banquise, donnée par deux coordonnées.
#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq, PartialOrd, Ord)]
pub struct Position(pub c_int, pub c_int);


/// Alien à capturer durant la mission.
#[derive(Clone, Copy, Debug, PartialEq, PartialOrd)]
pub struct AlienInfo {
  /// Position de l'alien
  pub pos: Position,
  /// Nombre de points obtenus pour la capture de cet alien
  pub points_capture: c_int,
  /// Tour où l'alien débarque sur la banquise
  pub tour_invasion: c_int,
  /// Nombre de tours où l'alien reste sur la banquise
  pub duree_invasion: c_int,
  /// Nombre de tours de la capture en cours, l'alien est considéré comme capturé si la variable atteint NB_TOURS_CAPTURE
  pub capture_en_cours: c_int,
}


/// Action de déplacement représentée dans l'historique.
#[derive(Clone, Debug, PartialEq, PartialOrd)]
pub struct ActionHist {
  /// Type de l'action
  pub atype: ActionType,
  /// Numéro de l'agent concerné par l'action
  pub id_agent: c_int,
  /// Direction visée par l'agent durant le déplacement
  pub dir: Direction,
}


/// Déplace l'agent ``id_agent`` d'une case dans la direction choisie.
pub fn deplacer(id_agent: c_int, dir: Direction) -> Erreur
{
  unsafe { ffi::deplacer(id_agent.to_ffi(), dir.to_ffi()).to_api() }
}

/// Propulse l'agent ``id_agent`` dans la direction choisie jusqu'à ce qu'il heurte un obstacle, c'est-à-dire soit un mur soit un autre agent.
pub fn glisser(id_agent: c_int, dir: Direction) -> Erreur
{
  unsafe { ffi::glisser(id_agent.to_ffi(), dir.to_ffi()).to_api() }
}

/// L'agent ``id_agent`` pousse tout autre agent se trouvant sur la case adjacente dans la direction indiquée. Ce dernier est propulsé jusqu'à ce qu'il rencontre un obstacle, c'est-à-dire soit un mur soit un autre agent.
pub fn pousser(id_agent: c_int, dir: Direction) -> Erreur
{
  unsafe { ffi::pousser(id_agent.to_ffi(), dir.to_ffi()).to_api() }
}

/// Affiche le drapeau spécifié sur la case indiquée.
pub fn debug_afficher_drapeau(pos: Position, drapeau: DebugDrapeau) -> Erreur
{
  unsafe { ffi::debug_afficher_drapeau(pos.to_ffi(), drapeau.to_ffi()).to_api() }
}

/// Renvoie le nombre de points d'action de l'agent ``id_agent`` restants pour le tour. Si le numéro d'agent est invalide, la fonction renvoie -1.
pub fn points_action_agent(id_agent: c_int) -> c_int
{
  unsafe { ffi::points_action_agent(id_agent.to_ffi()).to_api() }
}

/// Renvoie le plus court chemin entre deux positions de la banquise sous la forme d'une suite de direction à emprunter. Ce chemin ne contient pas de glissade, uniquement des déplacements simples. Si la position est invalide ou que le chemin n'existe pas, le chemin renvoyé est vide.
pub fn chemin(pos1: Position, pos2: Position) -> Vec<Direction>
{
  unsafe { ffi::chemin(pos1.to_ffi(), pos2.to_ffi()).to_api() }
}

/// Renvoie le type d'une case donnée.
pub fn type_case(pos: Position) -> CaseType
{
  unsafe { ffi::type_case(pos.to_ffi()).to_api() }
}

/// Renvoie le numéro du joueur à qui appartient l'agent sur la case indiquée. Renvoie -1 s'il n'y a pas d'agent ou si la position est invalide.
pub fn agent_sur_case(pos: Position) -> c_int
{
  unsafe { ffi::agent_sur_case(pos.to_ffi()).to_api() }
}

/// Indique si un alien se trouve sur une case donnée. Renvoie vrai si l'alien est en train d'envahir la banquise et qu'il n'a pas encore été capturé. Renvoie faux autremement, ou si la position est invalide.
pub fn alien_sur_case(pos: Position) -> bool
{
  unsafe { ffi::alien_sur_case(pos.to_ffi()).to_api() }
}

/// Indique la position de l'agent sur la banquise désigné par le numéro ``id_agent`` appartenant au joueur ``id_joueur``. Si la description de l'agent est incorrecte, la position (-1, -1) est renvoyée.
pub fn position_agent(id_joueur: c_int, id_agent: c_int) -> Position
{
  unsafe { ffi::position_agent(id_joueur.to_ffi(), id_agent.to_ffi()).to_api() }
}

/// Renvoie la description d'un alien en fonction d'une position donnée. Si l'alien n'est pas présent sur la carte, ou si la position est invalide, tous les membres de la structure ``alien_info`` renvoyée sont initialisés à -1.
pub fn info_alien(pos: Position) -> Option<AlienInfo>
{
  let ret = unsafe { ffi::info_alien(pos.to_ffi()).to_api() };
  if ret.points_capture == -1 { None } else { Some(ret) }
}

/// Renvoie la liste de tous les aliens présents durant la partie.
pub fn liste_aliens() -> Vec<AlienInfo>
{
  unsafe { ffi::liste_aliens().to_api() }
}

/// Renvoie la liste des actions effectuées par l’adversaire durant son tour, dans l'ordre chronologique. Les actions de débug n'apparaissent pas dans cette liste.
pub fn historique() -> Vec<ActionHist>
{
  unsafe { ffi::historique().to_api() }
}

/// Renvoie le score du joueur ``id_joueur``. Renvoie -1 si le joueur est invalide.
pub fn score(id_joueur: c_int) -> c_int
{
  unsafe { ffi::score(id_joueur.to_ffi()).to_api() }
}

static mut MOI: i32 = 0;
/// Renvoie votre numéro de joueur.
pub fn moi() -> c_int
{
  unsafe {
    if MOI == 0 {
        MOI = ffi::moi()
    }
    MOI
  }
}

static mut TOI: i32 = 0;
/// Renvoie le numéro de joueur de votre adversaire.
pub fn toi() -> c_int
{
  unsafe {
    if TOI == 0 {
        TOI = ffi::adversaire()
    }
    TOI
  }
}

/// Annule la dernière action. Renvoie faux quand il n'y a pas d'action à annuler ce tour-ci.
pub fn annuler() -> bool
{
  unsafe { ffi::annuler().to_api() }
}

/// Retourne le numéro du tour actuel.
pub fn tour_actuel() -> c_int
{
  unsafe { ffi::tour_actuel().to_api() }
}

/// Affiche le contenu d'une valeur de type case_type
pub fn afficher_case_type(v: CaseType)
{
  unsafe { ffi::afficher_case_type(v.to_ffi()).to_api() }
}

/// Affiche le contenu d'une valeur de type direction
pub fn afficher_direction(v: Direction)
{
  unsafe { ffi::afficher_direction(v.to_ffi()).to_api() }
}

/// Affiche le contenu d'une valeur de type erreur
pub fn afficher_erreur(v: Erreur)
{
  unsafe { ffi::afficher_erreur(v.to_ffi()).to_api() }
}

/// Affiche le contenu d'une valeur de type action_type
pub fn afficher_action_type(v: ActionType)
{
  unsafe { ffi::afficher_action_type(v.to_ffi()).to_api() }
}

/// Affiche le contenu d'une valeur de type debug_drapeau
pub fn afficher_debug_drapeau(v: DebugDrapeau)
{
  unsafe { ffi::afficher_debug_drapeau(v.to_ffi()).to_api() }
}

/// Affiche le contenu d'une valeur de type position
pub fn afficher_position(v: Position)
{
  unsafe { ffi::afficher_position(v.to_ffi()).to_api() }
}

/// Affiche le contenu d'une valeur de type alien_info
pub fn afficher_alien_info(v: AlienInfo)
{
  unsafe { ffi::afficher_alien_info(v.to_ffi()).to_api() }
}

/// Affiche le contenu d'une valeur de type action_hist
pub fn afficher_action_hist(v: ActionHist)
{
  unsafe { ffi::afficher_action_hist(v.to_ffi()).to_api() }
}

/// Fonction appelée au début de la partie.
#[no_mangle]
pub unsafe extern "C" fn partie_init()
{

  STATE = Some(super::partie_init())
}

/// Fonction appelée à chaque tour.
#[no_mangle]
pub unsafe extern "C" fn jouer_tour()
{
  super::jouer_tour(&mut STATE.as_mut().unwrap()).to_ffi()
}

/// Fonction appelée à la fin de la partie.
#[no_mangle]
pub unsafe extern "C" fn partie_fin()
{
  super::partie_fin(&mut STATE.as_mut().unwrap()).to_ffi()
}
