#include "utils.h"
#include <queue>
#include <iomanip>

bool isBuildable(position p)
{
    if(p.x <= 0 || p.x >= TAILLE_TERRAIN-1) return false;
    if(p.y <= 0 || p.y >= TAILLE_TERRAIN-1) return false;
    if(est_pulsar(p)) return false;
    return true;
}

bool inMyBase(position p)
{
    return type_case(p) == BASE && proprietaire_base(p) == moi();
}

bool connectedToMyBase(position p)
{
    for(position dir : DIRECTIONS)
    {
        if(inMyBase(p+dir)) return true;
    }
    return false;
}

enum WeaknessState
{
    NotComputed,
    Weak,
    Strong
};
std::vector<std::vector<WeaknessState> > weaknessMap;
std::vector<std::vector<int> > depthMap;

int computeWeakness(position pos, int depth, position parent)
{
    if(weaknessMap[pos.x][pos.y] == Weak || weaknessMap[pos.x][pos.y] == Strong) return depthMap[pos.x][pos.y];
    weaknessMap[pos.x][pos.y] = Weak;
    depthMap[pos.x][pos.y] = depth;

    int minDepthSafe = depth+1;
    for(position dir : DIRECTIONS)
    {
        if(pos+dir == parent) continue;
        if(est_tuyau(pos+dir) || inMyBase(pos+dir))
        {
            minDepthSafe = std::min(minDepthSafe, computeWeakness(pos+dir, depth+1, pos));
        }
    }

    if(minDepthSafe <= depth) weaknessMap[pos.x][pos.y] = Strong;
    return minDepthSafe;
}

void makeWeakList()
{
    weaknessMap = std::vector<std::vector<WeaknessState>>(TAILLE_TERRAIN, std::vector<WeaknessState>(TAILLE_TERRAIN, NotComputed));
    depthMap = std::vector<std::vector<int>>(TAILLE_TERRAIN, std::vector<int>(TAILLE_TERRAIN, 0));
    for(position p : ma_base()) computeWeakness(p,0,p);

    for(int x = 0 ; x < TAILLE_TERRAIN ; x++)
    {
        std::cout << std::right;
        for(int y = 0 ; y < TAILLE_TERRAIN ; y++)
        {
            std::cout << std::setw(2) << depthMap[x][y] << " ";
        }
        std::cout << std::endl;
    }
    std::cout << std::endl;

    for(int x = 0 ; x < TAILLE_TERRAIN ; x++)
    {
        for(int y = 0 ; y < TAILLE_TERRAIN ; y++)
        {
            if(weaknessMap[x][y] == Weak) std::cout << "W ";
            else if(weaknessMap[x][y] == Strong) std::cout << "S ";
            else std::cout << ". ";
        }
        std::cout << std::endl;
    }
}


int distToBase(position from)
{
    std::vector<position> toExplore;
    std::vector<position> toExploreNext;
    std::vector<std::vector<bool>> visited(TAILLE_TERRAIN, std::vector<bool>(TAILLE_TERRAIN, false));
    toExplore.push_back(from);

    int depth = 1;
    while(!toExplore.empty())
    {
        for(position pos : toExplore)
        {
            if(visited[pos.x][pos.y]) continue;
            visited[pos.x][pos.y] = true;

            for(position dir : DIRECTIONS)
            {
                position newPos = pos+dir;
                if(type_case(newPos) == BASE && proprietaire_base(newPos) == moi()) return depth;
                if(!isBuildable(newPos)) continue;
                if(est_debris(newPos)) continue;
                toExploreNext.push_back(newPos);
            }
        }

        toExplore = toExploreNext;
        toExploreNext.clear();
        depth++;
    }

    return INFINITY;
}


struct DijkNode
{
    std::vector<position> path;
    int cost;
    bool operator<(const DijkNode& o) const { return cost > o.cost; }
};

std::pair<std::vector<position>, int> dijkPath(position from, CheckPosFunc canBeEndPos, EvalPosFunc computeCost, CheckPosFunc validatePos)
{
    std::vector<std::vector<bool>> visited(TAILLE_TERRAIN, std::vector<bool>(TAILLE_TERRAIN, false));
    std::priority_queue<DijkNode> queue;
    queue.push(DijkNode{std::vector<position>(1,from),0});

    while(!queue.empty())
    {
        DijkNode node = queue.top();
        queue.pop();
        position currentPos = node.path.back();

        if(visited[currentPos.x][currentPos.y]) continue;
        visited[currentPos.x][currentPos.y] = true;
        if(canBeEndPos(currentPos)) return std::make_pair(node.path, node.cost);

        for(position dir : DIRECTIONS)
        {
            position newPos = currentPos+dir;

            if(!validatePos(newPos)) continue;

            std::vector<position> path = node.path;
            path.push_back(newPos);
            int cost = node.cost + computeCost(newPos);
            queue.push(DijkNode{path, cost});
        }
    }

    return std::make_pair(std::vector<position>(), INFINITY);
}

std::pair<std::vector<position>, int> bestPath(position from)
{
    auto computeCost = [](position p)
    {
        if(est_tuyau(p))
        {
            if(weaknessMap[p.x][p.y] == Weak) return 8;
            else return 5;
        }
        if(est_debris(p)) return 30;
        else return 10;
    };
    return dijkPath(from, connectedToMyBase, computeCost, isBuildable);
}

std::pair<std::vector<position>, int> upgradePath(position from)
{
    auto computeCost = [](position p)
    {
        if(weaknessMap[p.x][p.y] == Weak) return 8;
        else return 5;
    };
    return dijkPath(from, connectedToMyBase, computeCost, est_tuyau);
}

std::vector<position> choosePath(CheckPosFunc startCheck, PathFunc pathFunc, CheckPathFunc alreadyConstructed)
{
    std::vector<position> chosenPath;
    double bestScore = 0;
    for(position pulsar : liste_pulsars())
    {
        pulsar_info info = info_pulsar(pulsar);
        if(info.pulsations_restantes == 0) continue;
        double pulsar_value = info.puissance*info.pulsations_restantes/info.periode;

        for(position dir : DIRECTIONS)
        {
            position pos = pulsar+dir;
            if(!startCheck(pos)) continue;

            std::pair<std::vector<position>, int> optiPathStruct = pathFunc(pos);
            std::vector<position> optiPath = optiPathStruct.first;
            int optiLength = optiPathStruct.second;

            if(optiLength == INFINITY) continue;
            if(alreadyConstructed(optiPath)) continue;

            int distReal = distToBase(pos);
            int pipeVal = optiLength+2*distReal;
            double score = pulsar_value/pipeVal/pipeVal;
            if(score >= bestScore)
            {
                bestScore = score;
                chosenPath = optiPath;
            }
        }
    }

    return chosenPath;
}
