#include "gamestate.hpp"

// GameMap
bool GameMap::isValid(position pos) {
	return pos.ligne >= 0 && pos.colonne >= 0 && pos.ligne < MAP_SIZE && pos.colonne < MAP_SIZE;
}

bool GameMap::isEmpty(position pos) {
	return this->isValid(pos) && this->map[pos.ligne][pos.colonne].type == CELL_EMPTY;
}

bool GameMap::isWall(position pos) {
	if (!this->isValid(pos) || this->get(pos).type == CELL_WALL) {
		return true;
	}
	return false;
}
bool GameMap::isEnemy(position pos) {
	if (this->isValid(pos) && this->get(pos).type == CELL_AGENT_ENEMY) {
		return true;
	}
	return false;
}

bool GameMap::canMyAgentGoTo(position pos, int agentId) {
	if (this->isValid(pos)) {
		auto& cell = this->map[pos.ligne][pos.colonne];
		if (cell.type == CELL_EMPTY || (cell.type == CELL_AGENT_MINE && cell.agentId == agentId)) {
			return true;
		}
	}
	return false;
}

CellType& GameMap::get(position pos) {
	return this->map[pos.ligne][pos.colonne];
}

int GameMap::tryGetEnemyAt(position pos) {
	if (this->isValid(pos)) {
		auto& cell = this->get(pos);
		if (cell.type == CELL_AGENT_ENEMY) {
			return cell.agentId;
		}
	}
	cerr << "***** Agent enemy not found *****\n";
	return -1;
}

// VirtGameState

GameMap VirtGameState::baseMap;

void VirtGameState::setFromNow() {
	this->turnId = tour_actuel();
	for (int iAgent = 0; iAgent < NB_AGENTS; iAgent++) {
		this->myAgents[iAgent] = {position_agent(moi(), iAgent), NB_POINTS_ACTION};
		this->enemyAgents[iAgent] = {position_agent(adversaire(), iAgent), NB_POINTS_ACTION};
	}
	this->aliens = liste_aliens();
	this->nbAliens = this->aliens.size();
	this->myScore = score(moi());
	this->hisScore = score(adversaire());
}

void VirtGameState::setStateNewTurn() {
	this->map = VirtGameState::baseMap;
	for (int iAgent = 0; iAgent < NB_AGENTS; iAgent++) {
		this->map.get(this->myAgents[iAgent].pos) = {CELL_AGENT_MINE, iAgent};
		this->myAgents[iAgent].points = NB_POINTS_ACTION;
	}
	for (int iAgent = 0; iAgent < NB_AGENTS; iAgent++) {
		this->map.get(this->enemyAgents[iAgent].pos) = {CELL_AGENT_ENEMY, iAgent};
		this->enemyAgents[iAgent].points = NB_POINTS_ACTION;
	}
}

bool VirtGameState::isAlienCaptured(int iAlien) {
	return this->aliens[iAlien].capture_en_cours >= NB_TOURS_CAPTURE;
}

void VirtGameState::moveAgentTo(AgentState& agent, position destination) {
	if (destination != agent.pos) {
		swap(this->map.get(agent.pos), this->map.get(destination));
		agent.pos = destination;
	}
}

void VirtGameState::applyAction(ActionHist act) {
	AgentState& agent = this->myAgents[act.agentId];
	if (act.type == ACTION_POUSSER) {
		int enemy = this->map.tryGetEnemyAt(agent.pos + moves[act.direction]);
		if (enemy != -1) {
			this->applyAction({ACTION_GLISSER, enemy, act.direction});
		}
	} else if (act.type == ACTION_GLISSER) {
		this->moveAgentTo(agent, ApplyMoveTo(agent.pos, MOVE_GLISSE, act.direction, *this));
	} else {
		this->moveAgentTo(agent, ApplyMoveTo(agent.pos, MOVE_BASE, act.direction, *this));
	}
}

void VirtGameState::swapPlayers() {
	swap(this->myScore, this->hisScore);
	swap(this->myAgents, this->enemyAgents);
}

// Functions

position ApplyMoveTo(position pos, int type, int direction, VirtGameState& state) {
	if (type == MOVE_BASE) {
		position p2 = pos + moves[direction];
		if (state.map.isEmpty(p2)) {
			return p2;
		}
	} else {
		while (state.map.isEmpty(pos + moves[direction])) {
			pos = pos + moves[direction];
		}
	}
	return pos;
}

bool isCellOk(position pos) {
	return VirtGameState::baseMap.isEmpty(pos);
}

// Situation evaluation

double evalAgentOn(alien_info alien) {
	return (double)alien.points_capture
		* (1.- (conf::eval_perte_time
			* (double)(NB_TOURS_CAPTURE - alien.capture_en_cours)
			/ (double)NB_TOURS_CAPTURE))
		* conf::eval_coeff_positions;
}

double VirtGameState::evalState() {
	double val = (this->myScore - this->hisScore) * conf::eval_coeff_points;

	/* Points from aliens */
	for (auto& alien : this->aliens) {
		if (alien.tour_invasion <= this-> turnId && alien.capture_en_cours != NB_TOURS_CAPTURE &&
			alien.tour_invasion + alien.duree_invasion + NB_TOURS_CAPTURE - alien.capture_en_cours >= this->turnId) {
			for (auto& agent : this->myAgents) {
				if (agent.pos == alien.pos) {
					val += evalAgentOn(alien);
				}
			}
			for (auto& agent : this->enemyAgents) {
				if (agent.pos == alien.pos) {
					val -= evalAgentOn(alien);
				}
			}
		}
	}
	return val;
}