/// This file has been generated, if you wish to
/// modify it in a permanent way, please refer
/// to the script file : gen/generator_cxx.rb
#include "prologin.hh"
#include <ctime>
#include <iostream>

#define AGENT_MOI 3
#define AGENT_LUI 4
// Flags pour move_t
#define POUSSER_FLAG 2048
#define NULLMOVE_FLAG 4096
#define CAP_1_FLAG 8192
#define CAP_2_FLAG 16384
#define CAP_3_FLAG 32768

// Flags pour le djisktra
#define GLISSER_FLAG 2048

typedef int plateau[625];
typedef struct {
	int pos;
	int start;
	int cout;
	int flags; // Les premiers bits du flag peuvent contenir de l'information, comme la direction a pousser
} move_t;

alien_info aliens_pos[625];
std::vector<alien_info> aliens_list;

typedef struct {
	plateau p;
	
	int agents_moi[4];
	int agents_lui[4];
	int agents_PA[4];
	
	plateau cap;
	int tour;
} gamestate;

direction directions[4] = {NORD, EST, SUD, OUEST};

void dist_to_cases(plateau& pla, int pos_start, plateau& result, plateau& dual_graph);

/// Fonctions pratiques

int from_position(position& pos) {
	return pos.ligne * 25 + pos.colonne;
}

int abs(int x) {
	return x < 0 ? -x : x;
}

float fabs(float x) {
	return x < 0 ? -x : x;
}

int l1_dist(int pos, int pos2) {
	return abs(pos%25 - pos2%25) + abs(pos/25-pos2/25);
}

bool est_capturable(alien_info& alien, int tour, int cap) {
	return cap < NB_TOURS_CAPTURE && (alien.tour_invasion <= tour && alien.duree_invasion + alien.tour_invasion + cap - 3 >= tour);
}

bool cmp_plateau(plateau& p1, plateau& p2) {
	for(int i = 0 ; i < 625 ; i++) {
		if(p1[i] != p2[i])
			return false;
	}
	return true;
}

direction diff_to_direction(int diff) {
	if(diff > 0) {
		if(diff < 25)
			return EST;
		return SUD;
	} else {
		if(diff > -25) {
			return OUEST;
		}
		return NORD;
	}
}

inline bool est_safe(gamestate& g, int pos, int friends) {
	int c = pos%25;
	int l = pos/25;
	int v = 0;
	int h = 0;
	// positions adjaceantes
	if(c != 0) {
		h += g.p[pos-1] == MUR || g.p[pos-1] == friends;
	}
	if(c != 24) {
		h += g.p[pos+1] == MUR || g.p[pos+1] == friends ;
	}
	if(l != 0) {
		v += g.p[pos-25] == MUR || g.p[pos-25] == friends;
	}
	if(l != 24) {
		v += g.p[pos+25] == MUR || g.p[pos+25] == friends;
	}
	return h > 0 && v > 0;
}

int pousser_res(plateau& p, int pos, direction dir) {
	int c = pos % 25;
	int l = pos / 25;
	if(dir == EST) {
		int dec = 1;
		while(c+dec < 25 && p[pos+dec] <= 0) {
			dec += 1;
		}
		dec -= 1;
		return pos+dec;
	}
	if(dir == OUEST) {
		int dec = 1;
		while(c-dec >= 0&& p[pos-dec] <= 0) {
			dec += 1;
		}
		dec -= 1;
		return pos-dec;
	}
	if(dir == NORD) {
		int dec = 1;
		while(l-dec >= 0 && p[pos-25*dec] <= 0) {
			dec += 1;
		}
		dec -= 1;
		return pos-25*dec;
	}
	if(dir == SUD) {
		int dec = 1;
		while(l+dec < 25 && p[pos+25*dec] <= 0) {
			dec += 1;
		}
		dec -= 1;
		return pos+25*dec;
	}
	return -156;
}

std::vector<int> gen_sides(plateau& p, int pos) {
	std::vector<int> mv = std::vector<int>();
	int c = pos%25;
	int l = pos/25;
	
	// positions adjaceantes
	if(c != 0) {
		mv.push_back(pos-1);
	}
	if(c != 24) {
		mv.push_back(pos+1);
	}
	if(l != 0) {
		mv.push_back(pos-25);
	}
	if(l != 24) {
		mv.push_back(pos+25);
	}
	return mv;
}

void gen_mouvements(plateau& p, int pos, int* moves) {
	std::vector<int> mv = std::vector<int>();
	int c = pos%25;
	int l = pos/25;
	int ptr = 0;
	
	// positions adjaceantes
	if(c != 0) {
		moves[ptr++] = pos-1;
	}
	if(c != 24) {
		moves[ptr++] = pos+1;
	}
	if(l != 0) {
		moves[ptr++] = pos-25;
	}
	if(l != 24) {
		moves[ptr++] = pos+25;
	}
	//glissade a droite
	int dec = 1;
	while(c+dec < 25 && p[pos+dec] <= 0) {
		dec += 1;
	} 
	dec -= 1;
	if(dec >= 3) {
		moves[ptr++] = (pos+dec) | GLISSER_FLAG;
	}
	
	//glissade a gauche
	dec = 1;
	while(c-dec >= 0 && p[pos-dec] <= 0) {
		dec += 1;
	} 
	dec -= 1;
	if(dec >= 3) {
		moves[ptr++] = (pos-dec) | GLISSER_FLAG;
	}
	
	//glissade en haut
	dec = 1;
	while(l-dec >= 0 && p[pos-25*dec] <= 0) {
		dec += 1;
	} 
	dec -= 1;
	if(dec >= 3) {
		moves[ptr++] = (pos-25*dec) | GLISSER_FLAG;
	}
	
	//glissade en bas
	dec = 1;
	while(l+dec < 25 && p[pos+25*dec] <= 0) {
		dec += 1;
	} 
	dec -= 1;
	if(dec >= 3) {
		moves[ptr++] = (pos+25*dec) | GLISSER_FLAG;
	}
	moves[ptr] = -1;
}

/// Fonctions d'affichage

void afficher_coup(move_t mv) {
	printf("%d-%d %d-%d %d %d\n", mv.start%25, mv.start/25, mv.pos%25, mv.pos/25, mv.cout, mv.flags);
}

void afficher_raw(plateau& p) {
	for(int y = 0 ; y < 25 ; y++) {
		for(int x = 0 ; x < 25 ; x++) {
			int v = p[y*25+x];
			if(v < 10)
				printf("%d   ", v);
			else if(v < 100)
				printf("%d  ", v);
			else if(v < 1000)
				printf("%d ", v);
			else
				printf("%d..", v/100);
		}
		printf("\n");
	}
}

void afficher_plateau(plateau& p, bool afficher_aliens, int tour) {
	for(int y = 0 ; y < 25 ; y++) {
		for(int x = 0 ; x < 25 ; x++) {
			char c;
			int v = p[y*25+x];
			switch(v) {
				case LIBRE:
					c = '.';
					break;
				case MUR:
					c = '*';
					break;
				case AGENT_LUI:
					c = 'E';
					break;
				case AGENT_MOI:
					c = 'M';
					break;
				default:
					c = ' ';
					break;
			}
			if(c == ' ') {
				printf("ERREUR: Carte corrompue\n");
			}
			if(afficher_aliens) {
				alien_info alien = aliens_pos[y*25+x];
				if(est_capturable(alien, tour, alien.capture_en_cours)) {
					if(c == '.') {
						c = 'a';
					}
				}
			}
			printf("%c", c);
		}
		printf("\n");
	}
}

void afficher_state(gamestate& state) {
	afficher_plateau(state.p, true, state.tour);
	for(int i = 0 ; i < 4 ; i++) {
		printf("%d ", state.agents_moi[i]);
	}
	printf("\n");
	for(int i = 0 ; i < 4 ; i++) {
		printf("%d ", state.agents_lui[i]);
	}
	printf("\n");
	for(int i = 0 ; i < 4 ; i++) {
		printf("%d ", state.agents_PA[i]);
	}
	printf("\n");
	for(int i = 0 ; i < 625 ; i++) {
		if(state.cap[i] != 0) {
			printf("cap %d -> %d\n", i, state.cap[i]);
		}
	}
	printf("%d\n", state.tour);
}

/**
 * Permet d'appliquer un coup "move_t" a un etat. Cela etait initialiement prevu dans l'optique
 * d'appliquer plusieurs coup puis de les annuler, mais au final il fut impossible d'aller plus loin
 * que 1 de profondeur. (due a la lenteur de la fonction de score).
 */
void appliquer_move(gamestate& g, move_t& mv) {
	// Deplacement de l'agent sur la carte
	int val = g.p[mv.start];
	g.p[mv.start] = 0;
	g.p[mv.pos] = val;
	// Deplacement de l'agent dans les variables de positions
	if(val == AGENT_MOI) {
		for(int i = 0 ; i < 4 ; i++) {
			if(g.agents_moi[i] == mv.start) {
				g.agents_moi[i] = mv.pos;
				g.agents_PA[i] -= mv.cout;
			}
		}
	}
	if(val == AGENT_LUI) {
		for(int i = 0 ; i < 4 ; i++) {
			if(g.agents_lui[i] == mv.start) {
				g.agents_lui[i] = mv.pos;
				g.agents_PA[i] -= mv.cout;
			}
		}
	}
	
	// Si on a le flag "pousser", on recupere la position a deplacer, puis on calcule la direction et on effectue le poussement.
	if((mv.flags & POUSSER_FLAG) > 0) {
		int to_push = mv.flags & 1023; // Les premiers bits correspondent a la position de l'objet a pousser
		int res = pousser_res(g.p, to_push, diff_to_direction(to_push - mv.pos)); // On calcule la destination de l'objet pousse
		int pushed_id = g.p[to_push];
		g.p[to_push] = 0;
		g.p[res] = pushed_id;
		if(res != to_push) { // Si l'objet a effectivement ete deplace, la capture est remis a zero.
			if(g.cap[to_push] == 1) {// On va enregistrer l'ancien etat de la capture (1 ou 2) dans des flags correspondant.
				mv.flags |= CAP_1_FLAG;
			} else if(g.cap[to_push] == 2) {
				mv.flags |= CAP_2_FLAG;
			} else if(g.cap[to_push] == 3) {
				mv.flags |= CAP_3_FLAG;
			} else if(g.cap[to_push] > 2) {
				printf("AH ! Cap corrupted.\n");
			}
			g.cap[to_push] = 0;
		}
		
		if(pushed_id == AGENT_MOI) { // On deplace aussi la variable de l'agent affecte
			for(int i = 0 ; i < 4 ; i++) {
				if(g.agents_moi[i] == to_push) {
					g.agents_moi[i] = res;
				}
			}
		}
		if(pushed_id == AGENT_LUI) {
			for(int i = 0 ; i < 4 ; i++) {
				if(g.agents_lui[i] == to_push) {
					g.agents_lui[i] = res;
				}
			}
		}
	}
}

/**
 * Fonction reciproque de "appliquer_move", elle effectue precisemment le code inverse, en remettant les choses a leur places.
 * On remarque que cela signifie que "move" doit contenir toute les informations concernant le changement d'etat.
 */
void annuler_move(gamestate& g, move_t& mv) {
	// Deplacement de l'agent
	int val = g.p[mv.pos];
	g.p[mv.pos] = 0;
	g.p[mv.start] = val;
	if(val == AGENT_MOI) {
		for(int i = 0 ; i < 4 ; i++) {
			if(g.agents_moi[i] == mv.pos) {
				g.agents_moi[i] = mv.start;
				g.agents_PA[i] += mv.cout;
			}
		}
	}
	if(val == AGENT_LUI) {
		for(int i = 0 ; i < 4 ; i++) {
			if(g.agents_lui[i] == mv.pos) {
				g.agents_lui[i] = mv.start;
				g.agents_PA[i] += mv.cout;
			}
		}
	}
	// On remet l'agent poussee
	if((mv.flags & POUSSER_FLAG) > 0) {
		int to_push = mv.flags & 1023;
		direction dir = diff_to_direction(to_push - mv.pos);
		int res = pousser_res(g.p, to_push, dir); // Calcul de la localisation de l'objet precedemment poussee
		if((mv.flags & CAP_1_FLAG) > 0) { // Si l'agent etait en train de cap, on remet les variables de cap.
			g.cap[to_push] = 1;
		}
		if((mv.flags & CAP_2_FLAG) > 0) {
			g.cap[to_push] = 2;
		}
		if((mv.flags & CAP_3_FLAG) > 0) {
			g.cap[to_push] = 3;
		}
		if(g.p[res] > 2) {
			return;
		}
		// Etant donne qu'on a envoye un agent fantome pour voir ou il atterirait, 
		// il touchera l'agent precedemment deplace, il faut donc effectuer un mouvement 
		// de plus dans cette direction.
		if(dir == EST) {
			res += 1;
		}
		if(dir == OUEST) {
			res -= 1;
		}
		if(dir == NORD) {
			res -= 25;
		}
		if(dir == SUD) {
			res += 25;
		}
		int pushed_id = g.p[res]; // Enchangement des coordonnees
		g.p[res] = 0;
		g.p[to_push] = pushed_id;
		if(pushed_id == AGENT_MOI) {
			for(int i = 0 ; i < 4 ; i++) {
				if(g.agents_moi[i] == res) {
					g.agents_moi[i] = to_push;
				}
			}
		}
		if(pushed_id == AGENT_LUI) {
			for(int i = 0 ; i < 4 ; i++) {
				if(g.agents_lui[i] == res) {
					g.agents_lui[i] = to_push;
				}
			}
		}
	}
}

/**
 * Cette fonction permet d'appeller l'API pour le deplacement d'un agent a l'aide des graphs donne par le djisktra.
 * (n'inclue donc pas l'action de pousser par exemple, qui est gere par jouer_coup).
 */
int deplacer_agent_api(int agent_id, int start, int dst, plateau& dual_graph) {
	int len = 0;
	std::vector<int> pos_succ = std::vector<int>();
	
	// On traverse le graph dual pour trouver le chemin le plus court
	while(dst != start) {
		pos_succ.push_back(dst);
		len += 1;
		dst = dual_graph[dst];
	}
	
	// On effectue les deplacements en appellant l'API
	for(int i = 0 ; i < len ; i++) {
		int next = pos_succ.back();
		pos_succ.pop_back();
		int diff = next-start;
		//printf("I should be at %d-%d --> %d-%d diff: %d\n", start%25, start/25, next%25, next/25, diff);
		if(diff == 1) {
			int err = deplacer(agent_id, EST);
			if(err != OK) { printf("Err trying to move err code: %d\n", err); break; }
		} else if(diff == -1) {
			int err = deplacer(agent_id, OUEST);
			if(err != OK) { printf("Err trying to move err code: %d\n", err); break; }
		} else if(diff == 25) {
			int err = deplacer(agent_id, SUD);
			if(err != OK) { printf("Err trying to move err code: %d\n", err); break; }
		} else if(diff == -25) {
			int err = deplacer(agent_id, NORD);
			if(err != OK) { printf("Err trying to move err code: %d\n", err); break; }
		} else if(diff > 0 && diff < 25) {
			int err = glisser(agent_id, EST);
			if(err != OK) { printf("Err trying to move err code: %d\n", err); break; }
		} else if(diff < 0 && diff > -25) {
			int err = glisser(agent_id, OUEST);
			if(err != OK) { printf("Err trying to move err code: %d\n", err); break; }
		} else if(diff > 0) {
			int err = glisser(agent_id, SUD);
			if(err != OK) { printf("Err trying to move err code: %d\n", err); break; }
		} else  {
			int err = glisser(agent_id, NORD);
			if(err != OK) { printf("Err trying to move err code: %d\n", err); break; }
		}
		start = next;
	}
	// On renvoie la position effective (et non celle demandee). Pratique pour debugger.
	return start;
}

/**
 * Cette fonction est appelle lorsque le coup finale a ete decide. Elle effectue le lien entre
 * les structures de donnees utilise et l'API officielle.
 */
void jouer_coup(gamestate& g, move_t& mv) {
	// On regarde quel agent est cense effectuer l'action
	int agent_id = -1;
	for(int i = 0 ; i < 4 ; i++) {
		if(g.agents_moi[i] == mv.start) {
			agent_id = i;
			break;
		}
	}
	if(agent_id == -1) { // Si l'action est MOVE_NULL, il n'y aura pas d'agent associe
		return;
	}
	// On calcule les informations pour trouver le chemin le plus court afin de deplacer l'agent.
	plateau res, dual; 
	dist_to_cases(g.p, mv.start, res, dual);
	// On effectue le deplacement
	int end = deplacer_agent_api(agent_id, mv.start, mv.pos, dual);
	
	// Si l'agent n'est pas arrivee a destination.. Quelque chose s'est mal passe. On affiche des informations pour debug.
	if(end != mv.pos) {
		afficher_plateau(g.p, true, g.tour);
		afficher_coup(mv);
		for(int i = 0 ; i < 4 ; i++) {
			printf("%d ", g.agents_PA[i]);
		}
		printf(" %d\n", agent_id);
		printf("Something is wrong.\n");
	}

	if((mv.flags & POUSSER_FLAG) > 0) { // Si l'action inclue le fait de pousser, on le fait.
		int to_push = mv.flags & 1023; // Position de l'objet a pousser
		int res = pousser(agent_id, diff_to_direction(to_push - mv.pos));
		if(res != OK) { // Pousser a rate..
			printf("Something is wrong with POUSSER. err code: %d\n", res);
		}
	}
}

/**
 * L'algorithme principal de recherche de chemin. C'est donc simplement un djikstra avec quelques subtilitees...
 * 
 * 1. Il calcule la distance en prenant en compte les 8 PA par tour. 
 *    En effet, cela ne coute pas 3 PA de faire un glissement quand on a 1 PA restants mais 4 
 *    (car il faut attendre la fin du tour).
 * 
 * 2. La file de priorite qu'il utilise sacrifie de la memoire au prix des performances. 
 *    En effet, on appellera plusieurs milliers de fois cette fonction qui recalcule a chaque fois les distances vers toute les cases de la map.
 *    Toute la memoire est donc prealloue en faisant 2 suppositions.
 *     - Un chemin ne demandera jamais plus de 350 PA (raisonnable, car < 25*25/2) --> MAX_DIST
 *     - Il n'y aura jamais plus de 200 positions ayant la meme distance a l'agent initial. --> MAX_SAME_DIST
 *       Aussi raisonnable car la distance est similaire a manhattan (pas exact a cause des glissades).
 *       et MAX_SAME_DIST correspond a la taille maximale du perimetre, qui est autour de ~4*25/2 au maximum.
 *    Cela permet d'initialiser toute la memoire des le debut, et ne necessite donc pas de mettre a jour+allouer des listes, 
 *    mais simplement des pointeurs vers la queue des listes. (situee dans pointers)
 * 
 *    A noter que grace a la magie de cette methode, il n'y a pas besoin de liberer/nettoyer tout le tableau a chaque fois,
 *    mais simplement remettre les pointers a 0.
 * 
 *  Concernant les arguments, result contiendra les distances d'un point "pos_start" a chaque point atteignable de la map.
 *  Les murs/agents contiendront aussi une valeur qui correspond au min des valeurs autour d'eux. 
 *  (pratique pour avoir la distance a un alien qui a deja un agent sur lui)
 */

#define MAX_DIST 350
#define MAX_SAME_DIST 200

int prio_queue[MAX_DIST*MAX_SAME_DIST]; // Stack de priorite (revient au meme qu'une file dans notre cas)
int pointers[MAX_DIST]; // Pointeurs vers la queue

// Permet d'ajouter un element a la fin d'une queue 
inline void prio_push_back(int queue, int pos) {
	prio_queue[queue+pointers[queue]*MAX_DIST] = pos;
	pointers[queue] += 1;
}

// Permet d'enlever + recuperer un element a la fin de la queue
inline int prio_pop_back(int queue) {
	pointers[queue] -= 1;
	return prio_queue[queue+pointers[queue]*MAX_DIST];
}

// Simple djistra, comme explique plus haut.
void dist_to_cases(plateau& pla, int pos_start, plateau& result, plateau& dual_graph) {
	// On initialise le tableau de resultat a +inf (ici represente par 1000)
	for(int i = 0 ; i < 625 ; i++) {
		result[i] = 1000;
	}
	
	result[pos_start] = 0;
	
	prio_push_back(0, pos_start);
	int t = 0;
	int left = 1;
	
	// Tableau qui contiendra les mouvements, plus efficace qu'une liste (m'a fait gagner 2x en perfs)
	int moves[9];
		
	while(left > 0) {
		int p = prio_pop_back(t);
		gen_mouvements(pla, p, moves);
	
		// On itere sur les mouvements
		for(int m_id = 0 ; m_id < 8 ; m_id++) {
			int move = moves[m_id];
			if(move < 0)
				break;
			int t_arrival = 1 + result[p];
			if((move & GLISSER_FLAG) > 0) { // Glissade.. coute un peu plus cher, et depends des PA restants.
				int reste = 8-(result[p]%8);
				if(reste >= 3)
					reste = 0;
				t_arrival += 2 + reste;
				move -= GLISSER_FLAG;
			}
			if(pla[move] > 0) { // Deja qqun sur l'arrivee, on mets quand meme la distance (ainsi a la fin, cela correspondra au min des distances autour)
				if(t_arrival - 1 < result[move]) {
					result[move] = t_arrival - 1;
					dual_graph[move] = p;
				}
				continue;
			}
			if(t_arrival < result[move] && t_arrival < MAX_DIST) { // On met a jour la file de prio
				result[move] = t_arrival;
				dual_graph[move] = p;
				prio_push_back(t_arrival, move);
			}
		}
		left -= 1; // On a consomme l'element
		while(left == 0 && t < MAX_DIST - 1) { // On avance dans la file de prio pour recuperer les prochains elements a traiter
			t += 1;
			left = pointers[t];
		}
	}
}

/**
 * Fonction principale donc, qui calcule le score d'un etat de jeu donnee (positions des agents, points de capture etc.)
 * Voir description en haut du fichier pour son fonctionnement detaille, etant donne que c'est un des points majeurs de 
 * la strategie
 */

// Coefficient de points associe a la capture d'un alien (coefficient majeur, donne 95% du score d'un plateau)
float coeff_capture = 0.4f;
// Coefficient de distances a l'agent le plus proche
float coeff_dist_score = 0.001f;
// Coefficient de presence sur le plateau
float coeff_presence = 0.0006f;
// Pourcentage de points perdue si la position est poussable
float coeff_kickable = 0.7f;
// Coefficient de prediction d'arrivee des aliens
float coeff_oracle = 0.025f;

float score_state(gamestate& g) {
	// Ce qui prends la majorite du temps de la fonction de cout
	// Calcul des distances les plus courtes sur tout le plateau pour tout les agents de la map
	plateau dists_moi[4]; // distances pour mes agents
	plateau dists_lui[4]; // distances pour ses agents
	plateau dual; // inutilise apres, simplement necessaire pour le calcul
	for(int i = 0 ; i < 4 ; i++) {
		dist_to_cases(g.p, g.agents_moi[i], dists_moi[i], dual);
		dist_to_cases(g.p, g.agents_lui[i], dists_lui[i], dual);
	}
	
	// Calcul du coefficient de presence
	float presence = 0;
	for(alien_info& al : aliens_list) {
		int i = from_position(al.pos);
		int u = 0;
		int val = g.p[i];
		if(val == AGENT_LUI) { // La case lui appartient deja 
			u = -1;
		} else if(val == AGENT_MOI) {// La case m'appartient deja
			u = 1;
		} else { // La persone la plus proche possede cette case
			int min = 1000;
			u = 1;
			for(int j = 0 ; j < 4 ; j++) {
				if(dists_moi[j][i] < min) {
					min = dists_moi[j][i];
				}
			}
			for(int j = 0 ; j < 4 ; j++) {
				if(dists_lui[j][i] == min) {
					u = 0;
				}
				if(dists_lui[j][i] < min) {
					u = -1;
					break;
				}
			}
		}
		
		// Pondere par les points de capture et la distance a l'apparition
		if(al.tour_invasion + al.duree_invasion >= g.tour) {
			presence += u * al.points_capture / (1 + abs(al.tour_invasion + al.duree_invasion / 2 - g.tour));
		}
	}
	
	float capture_score = 0;
	// Calcul du coefficient de capture pour moi
	for(int i = 0 ; i < 4 ; i++) {
		alien_info al = aliens_pos[g.agents_moi[i]];
		
		if(est_capturable(al, g.tour, g.cap[g.agents_moi[i]])) { // l'alien est capturable
			#ifdef DEBUG
			printf("%d-%d ", g.agents_moi[i]%25, g.agents_moi[i]/25);
			#endif
			
			capture_score += al.points_capture * (1 + g.cap[g.agents_moi[i]]*g.cap[g.agents_moi[i]]); // On ajoute au score poids * cap^2
			bool coin = est_safe(g, from_position(al.pos), AGENT_MOI); // Si je suis safe avec l'aide d'un agent alie et des murs, je ne pourrais pas etre pousse
			
			#ifdef DEBUG
			printf("safe: %d gives %d", coin, al.points_capture * (1 + g.cap[g.agents_moi[i]]*g.cap[g.agents_moi[i]]));
			#endif
			
			if(!coin) { // Je ne suis pas safe.. mais il ne peut pas forcement me pousser
				for(int j = 0 ; j < 4 ; j++) {
					// Eh bien si.. un des agents adverses peut me pousser avant que je finisse de capturer
					// A noter que je verifie que l'agent pouvant me pousser ne vient pas d'un alien vallant plus que moi, auquel cas l'operation est negative pour lui,
					// donc on se fiche qu'il vienne.
					if(aliens_pos[g.agents_lui[j]].points_capture <= al.points_capture && dists_lui[j][g.agents_moi[i]] <= 3+8*(2-g.cap[g.agents_moi[i]])) { 
						
						#ifdef DEBUG
						//printf(" can be kicked -> -%f", al.points_capture * (1 + g.cap[g.agents_moi[i]]*g.cap[g.agents_moi[i]]) * coeff_kickable);
						#endif
						
						// Je peut etre pousse, je perds donc coeff_kickable pourcents des points de la capture
						capture_score -= al.points_capture * (1 + g.cap[g.agents_moi[i]]*g.cap[g.agents_moi[i]]) * coeff_kickable;
						break;
					}
				}
			}
			#ifdef DEBUG 
			printf("\n");
			#endif
			
		}
		#ifdef DEBUG 
		else {
			//printf("no alien %d %d %d cap %d\n", al.tour_invasion, al.duree_invasion, g.tour, g.cap[g.agents_moi[i]]);
		}
		#endif
	}
	
	// Calcul du score associe a l'apparition proche d'aliens
	float oracle_score = 0.0f;
	for(int i = 0 ; i < 4 ; i++) {
		alien_info al = aliens_pos[g.agents_moi[i]];
		if(al.tour_invasion > g.tour) {
			oracle_score += al.points_capture / (float)(al.tour_invasion - g.tour);
			bool coin = est_safe(g, from_position(al.pos), AGENT_MOI);
			//printf("%d-%d safe: %d\n", g.agents_moi[i]%25, g.agents_moi[i]/25, coin);
			if(!coin) {
				for(int j = 0 ; j < 4 ; j++) {
					if(dists_lui[j][g.agents_moi[i]] <= 3) {
						oracle_score -= (al.points_capture * coeff_kickable) / (float)(al.tour_invasion - g.tour);
						break;
					}
				}
			}
		}
		al = aliens_pos[g.agents_lui[i]];
		if(al.tour_invasion > g.tour) {
			oracle_score -= al.points_capture / (float)(al.tour_invasion - g.tour);
			bool coin = est_safe(g, from_position(al.pos), AGENT_LUI);
			if(!coin) {
				for(int j = 0 ; j < 4 ; j++) {
					if(dists_moi[j][g.agents_lui[i]] <= 3) {
						oracle_score += (al.points_capture * coeff_kickable) / (float)(al.tour_invasion - g.tour);
						break;
					}
				}
			}
		}
	}
	
	for(int i = 0 ; i < 4 ; i++) {
		alien_info al = aliens_pos[g.agents_lui[i]];
		
		if(est_capturable(al, g.tour, g.cap[g.agents_lui[i]])) {
			#ifdef DEBUG
			int pos = from_position(al.pos);
			//printf("enemy.. %d-%d substracts %d\n", pos%25, pos/25, al.points_capture * (1 + g.cap[g.agents_lui[i]]));
			#endif
			capture_score -= al.points_capture * (1 + g.cap[g.agents_lui[i]]);
		}
	}
	float dist_score = 0;
	
	for(alien_info& al : aliens_list) {
		int al_pos = from_position(al.pos);
		if(est_capturable(al, g.tour, 0) && !(g.p[al_pos] > 2 && g.cap[al_pos] == 2)) {
			int min_dist = 100;
			for(int i = 0 ; i < 4 ; i++) {
				if(g.agents_moi[i] != al_pos && est_capturable(aliens_pos[g.agents_moi[i]], g.tour, g.cap[g.agents_moi[i]]))
					continue;
				int t_d = dists_moi[i][al_pos];
				if(t_d < min_dist) {
					min_dist = t_d;
				}
			}
			int min_dist_e = 100;
			for(int i = 0 ; i < 4 ; i++) {
				if(g.agents_lui[i] != al_pos && est_capturable(aliens_pos[g.agents_lui[i]], g.tour, g.cap[g.agents_lui[i]]))
					continue;
				int t_d = dists_lui[i][al_pos];
				if(t_d < min_dist_e) {
					min_dist_e = t_d;
				}
			}
			if(min_dist != 100) {
				dist_score -= min_dist * 0.001f;
			}
			
			if(min_dist_e == 100 || min_dist == 100) {
				continue;
			}
			
			dist_score += ((min_dist_e/8 - min_dist / 8)*2 + (min_dist_e - min_dist)) * al.points_capture;
		}
	}
	#ifdef DEBUG
	printf("%f %f %f %f \n", oracle_score * coeff_oracle, capture_score * coeff_capture, dist_score * coeff_dist_score, presence * coeff_presence);
	#endif
	dist_score *= coeff_dist_score;
	if(fabs(dist_score) > 4) {
		if(dist_score > 0)
			dist_score = 4 + (dist_score - 4 ) / 10.0f;
		else
			dist_score = -4 + (dist_score + 4 ) / 10.0f;
	}
	return oracle_score * coeff_oracle + capture_score * coeff_capture + dist_score + presence * coeff_presence;
}

void init_plateau(plateau& p) {
	for(int i = 0 ; i < 625 ; i++) {
		p[i] = 0;
	}
}

void parse_ascii(gamestate& g) {
	int cnt = 0;
	int cnt2 = 0;
	
const char* a = ".........................\
..*...................*..\
.........................\
.........M**.**E.........\
........**.....**........\
.........................\
...M..*...........*..E...\
...*..**..*...*..**..*...\
......a...........a......\
.....*..*.......*..*.....\
*.......*.**.**.*.......*\
*..*...*...*.*...*...*..*\
............*............\
.....*...*.....*...*.....\
........*...*...*........\
........*.......*........\
...*.................*...\
M.......................E\
*...*...............*...*\
........***...***........\
*..*........*........*..*\
.**...................**.\
..*...................*..\
......*....*.*....*......\
......*M.........E*......";

	for(int i = 0 ; i < 625 ; i++) {
		char v = a[i];
		if(v == '.') {
			g.p[i] = LIBRE;
		}
		if(v == 'E') {
			g.p[i] = AGENT_LUI;
			g.agents_lui[cnt++] = i;
		}
		if(v == 'M') {
			g.p[i] = AGENT_MOI;
			g.agents_moi[cnt2++] = i;
		}
		if(v == '*') {
			g.p[i] = MUR;
		}
		g.cap[i] = 0;
	}
}
std::vector<move_t> gen_fallback_moves(gamestate& state, int agent) {
	std::vector<move_t> good_moves = std::vector<move_t>();
	
	move_t null_move;
	null_move.start = 0;
	null_move.pos = 0;
	null_move.flags = 0;
	null_move.cout = NULLMOVE_FLAG;
	good_moves.push_back(null_move);
	
	int pos = state.agents_moi[agent];
	if(state.agents_PA[agent] >= 3) {
		for(direction& dir : directions) {
			int res = pousser_res(state.p, pos, dir);
			if(res != pos) {
				move_t mv;
				mv.flags = 888;
				mv.pos = res;
				mv.start = pos;
				mv.cout = 3;
				good_moves.push_back(mv);
			}
		}
	}
	for(int& m : gen_sides(state.p, pos)) {
		if(state.p[m] <= 0) {
			move_t mv;
			mv.flags = 888;
			mv.pos = m;
			mv.start = pos;
			mv.cout = 1;
			good_moves.push_back(mv);
		}
	}
	plateau res, dual;
	dist_to_cases(state.p, pos, res, dual);
	for(int j = 0 ; j < 4 ; j++) {
		int lui_pos = state.agents_lui[j];
		
		for(int& m : gen_sides(state.p, lui_pos)) {
			if(res[m] > 3-8+state.agents_PA[agent] || (state.p[m] > 0 && m != pos))
				continue;
			int to_add = m;
			//printf("mv %d PA %d\n", to_add, state.agents_PA[i]);
			move_t mv;
			mv.flags = POUSSER_FLAG | lui_pos;
			mv.pos = to_add;
			mv.start = pos;
			mv.cout = res[to_add] + 5;
			good_moves.push_back(mv);
		}
	}
	return good_moves;
}
std::vector<move_t> gen_interesting_moves(gamestate& state, int side) {
	std::vector<move_t> good_moves = std::vector<move_t>();
	move_t null_move;
	null_move.start = 0;
	null_move.pos = 0;
	null_move.flags = 0;
	null_move.cout = NULLMOVE_FLAG;
	int LUI = (side == api_moi()) ? AGENT_LUI : AGENT_MOI;
	
	good_moves.push_back(null_move);

	for(int i = 0 ; i < 4 ; i++) {
		//printf("%d -> %d\n", i, state.agents_PA[i]);
		if(state.agents_PA[i] == 0)
			continue;
		int pos = (side == api_moi()) ? state.agents_moi[i] : state.agents_lui[i];
		plateau res, dual;
		dist_to_cases(state.p, pos, res, dual);
		
		for(alien_info& al : aliens_list) { // Alien
			int al_pos = from_position(al.pos);
			//printf("%d -> %d\n", al_pos, state.p[al_pos]);
			if(al.points_capture == 0 || state.p[al_pos] > 0)
				continue;
			int t_dist = res[al_pos]/8;
			if(est_capturable(al, state.tour+t_dist, state.cap[al_pos])) {
				int to_add = al_pos;
				int s = 0;
				while(res[to_add] > state.agents_PA[i]) {
					to_add = dual[to_add];
					s++;
				}
				move_t mv;
				mv.pos = to_add;
				mv.start = pos;
				mv.flags = s + 10*(to_add == al_pos);
				mv.cout = res[to_add];
				if(mv.pos != mv.start) {
					good_moves.push_back(mv);
				}
			} else if(al.tour_invasion > state.tour && al.tour_invasion < state.tour + 5 && res[al_pos] <= state.agents_PA[i]) {
				int to_add = al_pos;
				move_t mv;
				mv.pos = to_add;
				mv.start = pos;
				mv.flags = 711;
				mv.cout = res[to_add];
				if(mv.pos != mv.start) {
					good_moves.push_back(mv);
				}
			}
		}
		
		for(alien_info& al : aliens_list) { // A cote de lalien avec qqun dessus
			if(al.points_capture == 0)
				continue;
			int al_pos = from_position(al.pos);
			int t_dist = res[al_pos]/8; 
			
			if(est_capturable(al, state.tour+t_dist, state.cap[al_pos]) && state.p[al_pos] > 0) {
				for(int& m : gen_sides(state.p, al_pos)) {
					
					if(res[m] == 1000 || (state.p[m] > 0 && m != pos))
						continue;
					
					int to_add = m;
					//printf("mv %d PA %d\n", to_add, state.agents_PA[i]);
					while(res[to_add] > state.agents_PA[i]) {
						to_add = dual[to_add];
					}
					move_t mv;
					mv.flags = 0;
					if(state.p[al_pos] == LUI && res[m] <= 3-8+state.agents_PA[i]) { // On ne vient pas pour rien, on vient pour pousser si on est suffisamment pres
						//printf("%d %d\n", res[m], state.agents_PA[i]);
						mv.flags = POUSSER_FLAG | al_pos;
					}
					mv.pos = to_add;
					mv.start = pos;
					mv.cout = res[to_add] + (((mv.flags & POUSSER_FLAG) > 0) ? 5 : 0);
					if(mv.pos != mv.start || (mv.flags & POUSSER_FLAG) > 0) {
						good_moves.push_back(mv);
					}
				}
			}
		}
		for(int j = 0 ; j < 4 ; j++) { // A cote de qqun (pour le pousser)
			int lui_pos = (side == api_moi()) ? state.agents_lui[j] : state.agents_moi[j];
			
			for(int& m : gen_sides(state.p, lui_pos)) {
				if(res[m] > 3-8+state.agents_PA[i] || (state.p[m] > 0 && m != pos))
					continue;
				int to_add = m;
				//printf("mv %d PA %d\n", to_add, state.agents_PA[i]);
				move_t mv;
				mv.flags = POUSSER_FLAG | lui_pos;
				mv.pos = to_add;
				mv.start = pos;
				mv.cout = res[to_add] + 5;
				good_moves.push_back(mv);
			}
		}
	}
	if(good_moves.size() < 10) { // Tres peu de coup possible, on ajoute les simples deplacements
		for(int i = 0 ; i < 4 ; i++) {
			int pos = (side == api_moi()) ? state.agents_moi[i] : state.agents_lui[i];
			if(state.agents_PA[i] == 0)
				continue;
			if(state.agents_PA[i] >= 3) {
				for(direction& dir : directions) {
					int res = pousser_res(state.p, pos, dir);
					if(res != pos) {
						move_t mv;
						mv.flags = 888;
						mv.pos = res;
						mv.start = pos;
						mv.cout = 3;
						good_moves.push_back(mv);
					}
				}
			} else {
				for(int& m : gen_sides(state.p, pos)) {
					if(state.p[m] <= 0) {
						move_t mv;
						mv.flags = 888;
						mv.pos = m;
						mv.start = pos;
						mv.cout = 1;
						good_moves.push_back(mv);
					}
				}
			}
		}
	}
	return good_moves;
}

void charger_jeu(gamestate& g) {
	g.tour = api_tour_actuel();
	
	for(int i = 0 ; i < 625 ; i++) {
		position pos;
		pos.ligne = i/25;
		pos.colonne = i%25;
		case_type info = type_case(pos);
		g.p[i] = info;
		g.cap[i] = 0;
	}
	
	for(alien_info& inf : aliens_list) {
		if(alien_sur_case(inf.pos)) {
			g.cap[from_position(inf.pos)] = info_alien(inf.pos).capture_en_cours;
		} else if(g.tour >= inf.tour_invasion && g.tour < inf.tour_invasion+inf.duree_invasion) {
			g.cap[from_position(inf.pos)] = 3;
			debug_afficher_drapeau(inf.pos, DRAPEAU_VERT);
		}
	}
	
	for(int i = 0 ; i < 4 ; i++) {
		position agent_pos = position_agent(api_moi(), i);
		g.agents_moi[i] = from_position(agent_pos);
		g.p[g.agents_moi[i]] = AGENT_MOI;
		g.agents_PA[i] = 8;
	}
	for(int i = 0 ; i < 4 ; i++) {
		position agent_pos = position_agent(api_adversaire(), i);
		g.agents_lui[i] = from_position(agent_pos);
		g.p[g.agents_lui[i]] = AGENT_LUI;
	}
	
}

void cpy_plateau(plateau& src, plateau& dst) {
	for(int i = 0 ; i < 625 ; i++) {
		dst[i] = src[i];
	}
}

/// Fonction appelée au début de la partie.
void partie_init()
{
	printf("Init start..\n");
	for(int i = 0 ; i < 625 ; i++) {
		aliens_pos[i].tour_invasion = -1000;
		aliens_pos[i].duree_invasion = 0;
		aliens_pos[i].points_capture = 0;
		aliens_pos[i].duree_invasion = 0;
	}
	
	for(int i = 0 ; i < MAX_DIST ; i++) {
		pointers[i] = 0;
	}
	aliens_list = liste_aliens();
	for(alien_info& alien : aliens_list) {
		aliens_pos[from_position(alien.pos)] = alien;
	}
	printf("Init OK\n");
}

/// Fonction appelée à chaque tour.
void jouer_tour()
{
	
	gamestate state;
	#ifndef DEBUG
	charger_jeu(state);
	#endif
	
	#ifdef DEBUG
	parse_ascii(state);
	state.agents_moi[0] = 425 ;
	state.agents_moi[1] = 153 ;
	state.agents_moi[2] = 84 ;
	state.agents_moi[3] = 607 ;
	state.agents_lui[0] = 449 ;
	state.agents_lui[1] = 171 ;
	state.agents_lui[2] = 90 ;
	state.agents_lui[3] = 617 ;

	state.agents_PA[0] = 8;
	state.agents_PA[1] = 8;
	state.agents_PA[2] = 8;
	state.agents_PA[3] = 8;
	state.tour = 0;
	#endif
	
	for(alien_info& al : aliens_list) {
		if(al.tour_invasion == state.tour + 2) {
			debug_afficher_drapeau(al.pos, DRAPEAU_BLEU);
		}
		if(al.tour_invasion == state.tour + 1) {
			debug_afficher_drapeau(al.pos, DRAPEAU_ROUGE);
		}
	}
	printf("Tour %d --- \n", state.tour);
	auto time = std::clock();
	auto time2 = std::clock();
	int total = 32;
	move_t best_move;
	best_move.start = -1;
	int coups = 0;
	afficher_state(state);
	bool use_fallback = false;
	do {
		#ifdef DEBUG
		afficher_plateau(state.p, true, state.tour);
		#endif
		float max_score = -1000000;
		std::vector<move_t> moves = gen_interesting_moves(state, api_moi());
		int max_moves = (850 - 1000*(time2-time)/CLOCKS_PER_SEC) * 2;
		int mv_count = 0;
		#ifndef DEBUG
		if(moves.size() > max_moves && moves.size() > 200) {
			use_fallback = true;
			break;
		}
		#endif
		for(move_t& mv : moves) {
			plateau tmp, tmp2;
			cpy_plateau(state.p, tmp);
			appliquer_move(state, mv);
			float minscore = score_state(state);
			
			#ifdef DEBUG
			printf("score %f for ", minscore);
			afficher_coup(mv);
			#endif
			cpy_plateau(state.p, tmp2);
			annuler_move(state, mv);
			
			if(!cmp_plateau(tmp, state.p)) {
				printf("AH !!! Pas bien ca !!!\n");
				afficher_coup(mv);
				afficher_plateau(tmp, true, state.tour);
				printf("-\n");
				afficher_plateau(tmp2, true, state.tour);
				printf("-\n");
				afficher_plateau(state.p, true, state.tour);
			}
			
			
			if(minscore > max_score) {
				max_score = minscore;
				best_move = mv;
			}
			mv_count++;
		}
		if(best_move.start != -1) {
			jouer_coup(state, best_move);
			appliquer_move(state, best_move);
			total -= best_move.cout;
			printf("%d coups analyse\n", mv_count);
			coups += mv_count;
			printf("score - %f ", max_score);
			afficher_coup(best_move);
			time2 = std::clock();
		} else {
			break;
		}
		printf("---\n");
	} while((best_move.flags & NULLMOVE_FLAG) == 0 && total > 0 && (1000*(time2-time)/CLOCKS_PER_SEC) < 850);
	if(use_fallback) {
		for(int i = 0 ; i < 4 ; i++) {
			if(state.agents_PA[i] == 8 && !est_capturable(aliens_pos[state.agents_moi[i]], state.tour, state.cap[state.agents_moi[i]])) {
				std::vector<move_t> moves = gen_fallback_moves(state, i);
				float max_score = -10000;
				
				move_t best_move;
				best_move.start = -1;
				
				for(move_t& mv : moves) {
					appliquer_move(state, mv);
					float minscore = score_state(state);
					
					#ifdef DEBUG
					printf("score %f for ", minscore);
					afficher_coup(mv);
					#endif
					
					annuler_move(state, mv);
					
					if(minscore > max_score) {
						max_score = minscore;
						best_move = mv;
					}
				}
				if(best_move.start != -1) {
					jouer_coup(state, best_move);
					appliquer_move(state, best_move);
					printf("fallback move -- score - %f ", max_score);
					afficher_coup(best_move);
				} else {
					break;
				}
			}
		}
	}
	std::cout << 1000*(std::clock()-time)/CLOCKS_PER_SEC << "ms taken, so thats " << (1000*(std::clock()-time)/CLOCKS_PER_SEC) / (float)(coups) << "ms per eval" << std::endl;
}

/// Fonction appelée à la fin de la partie.
void partie_fin()
{

}

int main() {
	partie_init();
	return 0;
}
