#include <iostream>
#include <queue>
#include <cmath>

#include "prologin.hh"

#define FOR_EACH_CELL(i, j) for (int i = 0; i < TAILLE_MINE; i++) for (int j = 0; j < TAILLE_MINE; col++)
#define FOR_EACH_DWARF(d) for (int d = 0; d < NB_NAINS; d++)

// Coût pour changer de cible en cours de route
#define GOAL_CHANGE_COST 100.
// Coût lié à la distance de la cible
#define GOAL_DISTANCE_FACTOR 1.
// Coût lié au rendement du minerai cible
#define GOAL_VALUE_FACTOR -100.
// Coût lié au butin du nain cible
#define GOAL_DWARF_VALUE_FACTOR -1.
// Coût lié à la résistance du minerai cible
#define GOAL_RESISTANCE_FACTOR 100.
// Coût lié au butin total quand la cible est un minerai ou un nain
#define GOAL_ORE_WEALTH_FACTOR 100.
// Coût lié au butin total quand la cible est la taverne
#define GOAL_TAVERN_WEALTH_FACTOR -10.
// Correspondance entre les points d'action et les points de mouvement
#define ACTION_POINTS_FACTOR (1./COUT_MINER)

// Rayon à l'intérieur duquel les nains d'une même équipe sont considérés "groupés"
#define GROUP_RADIUS (NB_POINTS_DEPLACEMENT/COUT_DEPLACEMENT)

#define DEBUG(str) std::cout << #str << std::endl

#undef INFINITY
#define INFINITY 9999999.

enum goal_type { NONE, ORE, DWARF, TAVERN };

struct goal {
    goal_type type = NONE;
    position pos;
    minerai ore_info = {-1, -1};
    int dwarf = -1;
};

position group_position;
bool dwarf_isolated[NB_NAINS];
goal current_goal;
int total_wealth;

int height_above_ground[TAILLE_MINE][TAILLE_MINE];

position operator+(position p, direction d) {
    switch (d) {
    case GAUCHE:
        return { p.ligne, p.colonne == 0 ? 0 : p.colonne - 1 };
    case DROITE:
        return { p.ligne, p.colonne == TAILLE_MINE-1 ? TAILLE_MINE-1 : p.colonne + 1 };
    case HAUT:
        return { p.ligne == 0 ? 0 : p.ligne - 1, p.colonne };
    case BAS:
        return { p.ligne == TAILLE_MINE-1 ? TAILLE_MINE-1 : p.ligne + 1, p.colonne };
    default:
        return { -1, -1 };
    }
}
position& operator+=(position& p, direction d) { p = p + d; return p; }

void print_position(position p, const std::string& name = "") {
    if (!name.empty())
        std::cout << name << ": ";
    std::cout << "ligne " << p.ligne << ", colonne " << p.colonne << std::endl;
}

void team_flag(position p) {
    debug_afficher_drapeau(p, moi() == 1 ? DRAPEAU_ROUGE : DRAPEAU_BLEU);
}

void calculate_total_wealth() {
    total_wealth = 0;
    FOR_EACH_DWARF (dwarf) {
        nain dwarf_info = info_nain(moi(), dwarf);
        total_wealth += dwarf_info.butin;
    }
}

void calculate_heights() {
    for (int col = 0; col < TAILLE_MINE; col++) {
        int height = 0;
        for (int row = TAILLE_MINE - 1; row >= 0; row--) {
            height_above_ground[row][col] = height;
            if (type_case({row, col}) == LIBRE && nain_sur_case({row, col}) != adversaire())
                height++;
            else
                height = 0;
        }
    }
}

position landing_point(position p) {
    position lp = p;
    for (int i = 0; i < height_above_ground[p.ligne][p.colonne]; i++)
        lp += BAS;
    return lp;
}

position group_average_position(int player) {
    position average{0, 0};
    bool all_dead = true;
    FOR_EACH_DWARF (dwarf) {
        nain dwarf_info = info_nain(player, dwarf);
        if (dwarf_info.vie > 0) {
            all_dead = false;
            average.ligne += dwarf_info.pos.ligne;
            average.colonne += dwarf_info.pos.colonne;
        }
    }
    if (all_dead)
        return position_taverne(moi());
    else {
        average.ligne /= NB_NAINS;
        average.colonne /= NB_NAINS;
        return average;
    }
}

int distance_manhattan(const position& a, const position& b) {
    return std::abs(a.ligne - b.ligne) + std::abs(a.colonne - b.colonne);
}

int distance_squared(const position& a, const position& b) {
    return (a.ligne - b.ligne)*(a.ligne - b.ligne) + (a.colonne - b.colonne)*(a.colonne - b.colonne);
}

direction direction_to(position a, position b) {
    int x = b.colonne - a.colonne, y = b.ligne - a.ligne;
    if (x > std::abs(y))
        return DROITE;
    else if (-x > std::abs(y))
        return GAUCHE;
    else if (y > std::abs(x))
        return BAS;
    else
        return HAUT;
}

struct node {
    position pos;
    bool holding = false;
};

struct move {
    node target;
    bool release = false;
    bool hold = false;
    direction mine = ERREUR_DIRECTION;
    int strokes = 0;
    direction movement = ERREUR_DIRECTION;
};

double move_cost(move m) {
    double cost = 0.;
    if (m.release)
        cost += ACTION_POINTS_FACTOR * COUT_LACHER;
    if (m.hold)
        cost += ACTION_POINTS_FACTOR * COUT_AGRIPPER;
    if (m.mine != ERREUR_DIRECTION)
        cost += m.strokes * ACTION_POINTS_FACTOR * COUT_MINER;
    if (m.movement != ERREUR_DIRECTION) {
        if (m.target.holding) {
            if (corde_sur_case(m.target.pos))
                cost += COUT_ESCALADER_CORDE;
            else
                cost += COUT_ESCALADER;
        } else
            cost += COUT_DEPLACEMENT;
    }
    return cost;
}

std::vector<move> get_moves(node n) {
    std::vector<move> moves;
    if (n.holding) {
        position lp = landing_point(n.pos);
        if (lp.ligne - n.pos.ligne <= 4)
            moves.push_back({{lp, false}, true});
    } else
        moves.push_back({{n.pos, true}, false, true});
    for (direction d : {GAUCHE, DROITE, HAUT, BAS}) {
        move m;
        position p = n.pos + d;
        if (p == n.pos)
            continue;
        case_type ct = type_case(p);
        if (ct == OBSIDIENNE)
            continue;
        else if (ct == GRANITE) {
            m.mine = d;
            minerai ore_info = info_minerai(p);
            m.strokes = ore_info.resistance == -1 ? 1 : ore_info.resistance;
        }
        if (n.holding || d != BAS)
            m.movement = d;
        if (n.holding)
            m.target = {p, true};
        else {
            m.target = {landing_point(p), false};
            if (m.target.pos.ligne - p.ligne > 4)
                continue;
        }
        moves.push_back(m);
    }
    return moves;
}

double distance_cost[TAILLE_MINE][TAILLE_MINE][2];

bool operator<(const node& a, const node& b) {
    return distance_cost[a.pos.ligne][a.pos.colonne][a.holding] > distance_cost[b.pos.ligne][b.pos.colonne][a.holding];
}

std::deque<move> get_path(position start, bool holding, position end = {-1, -1}) {
    if (start == end)
        return {};
    node path_parent[TAILLE_MINE][TAILLE_MINE][2];
    move path_parent_move[TAILLE_MINE][TAILLE_MINE][2];
    bool visited[TAILLE_MINE][TAILLE_MINE][2];
    FOR_EACH_CELL (row, col) {
        for (bool h : {false, true}) {
            distance_cost[row][col][h] = INFINITY;
            visited[row][col][h] = false;
        }
    }
    distance_cost[start.ligne][start.colonne][holding] = 0.;
    std::priority_queue<node> open;
    open.push({start, holding});
    while (!open.empty()) {
        node current = open.top();
        open.pop();
        if (visited[current.pos.ligne][current.pos.colonne][current.holding])
            continue;
        visited[current.pos.ligne][current.pos.colonne][current.holding] = true;
        if (current.pos == end)
            break;
        for (move m : get_moves(current)) {
            node n = m.target;
            double candidate_cost = distance_cost[current.pos.ligne][current.pos.colonne][current.holding] + move_cost(m);
            if (candidate_cost < distance_cost[n.pos.ligne][n.pos.colonne][n.holding]) {
                distance_cost[n.pos.ligne][n.pos.colonne][n.holding] = candidate_cost;
                path_parent[n.pos.ligne][n.pos.colonne][n.holding] = current;
                path_parent_move[n.pos.ligne][n.pos.colonne][n.holding] = m;
            }
            open.push(n);
        }
    }
    if (end != position{-1, -1}) {
        for (bool h : {false, true}) {
            if (visited[end.ligne][end.colonne][h]) {
                std::deque<move> path;
                node n{end, h};
                do {
                    path.push_front(path_parent_move[n.pos.ligne][n.pos.colonne][n.holding]);
                    n = path_parent[n.pos.ligne][n.pos.colonne][n.holding];
                } while (n.pos != start || n.holding != holding);
                return path;
            }
        }
    }
    return {};
}

double goal_cost(goal g) {
    double cost = 0.;
    int value = 0, health = 0;
    nain dwarf_info;
    if (current_goal.type != NONE && distance_manhattan(g.pos, current_goal.pos) > 6)
        cost += GOAL_CHANGE_COST;
    int distance = distance_cost[g.pos.ligne][g.pos.colonne][false];
    cost += GOAL_DISTANCE_FACTOR * distance;
    switch (g.type) {
    case ORE:
        cost += GOAL_VALUE_FACTOR * g.ore_info.rendement;
        if (g.ore_info.resistance > NB_TOURS*NB_NAINS)
            cost += INFINITY;
        break;
    case DWARF:
        FOR_EACH_DWARF (dwarf) {
            dwarf_info = info_nain(adversaire(), dwarf);
            if (distance_manhattan(dwarf_info.pos, g.pos) <= GROUP_RADIUS) {
                value += dwarf_info.butin;
                health += dwarf_info.vie;
            }
        }
        cost += GOAL_DWARF_VALUE_FACTOR * value;
        cost += GOAL_RESISTANCE_FACTOR * health;
        cost += GOAL_ORE_WEALTH_FACTOR * total_wealth;
        break;
    case TAVERN:
        cost += GOAL_TAVERN_WEALTH_FACTOR * total_wealth;
        break;
    default:
        break;
    }
    return cost;
}

goal find_goal() {
    std::vector<goal> goals;
    size_t min_index = 0;
    double min_cost = INFINITY;
    for (position ore_pos : liste_minerais()) {
        goal g{ORE, ore_pos};
        g.ore_info = info_minerai(ore_pos);
        goals.push_back(g);
    }
    FOR_EACH_DWARF (dwarf) {
        goal g{DWARF, info_nain(adversaire(), dwarf).pos};
        g.dwarf = dwarf;
        goals.push_back(g);
    }
    goals.push_back({TAVERN, position_taverne(moi())});
    for (size_t i = 0; i < goals.size(); i++) {
        double cost = goal_cost(goals[i]);
        if (cost < min_cost) {
            min_index = i;
            min_cost = cost;
        }
    }
    return goals[min_index];
}

void partie_init() {
}

void jouer_tour() {
    calculate_total_wealth();
    calculate_heights();
    position average_pos = group_average_position(moi());
    FOR_EACH_DWARF (dwarf) {
        nain dwarf_info = info_nain(moi(), dwarf);
        dwarf_isolated[dwarf] = distance_squared(dwarf_info.pos, average_pos) > GROUP_RADIUS*GROUP_RADIUS;
    }
    FOR_EACH_DWARF (dwarf) {
        group_position = average_pos;
        int vie_max = 0;
        FOR_EACH_DWARF (dwarf2) {
            nain dwarf2_info = info_nain(moi(), dwarf2);
            if (dwarf2 != dwarf && dwarf2_info.vie > vie_max) {
                vie_max = dwarf2_info.vie;
                group_position = dwarf2_info.pos;
            }
        }
    }
    get_path(group_position, false);
    current_goal = find_goal();
    team_flag(current_goal.pos);
    FOR_EACH_DWARF (dwarf) {
        nain dwarf_info = info_nain(moi(), dwarf);
        position target;
        if (dwarf_isolated[dwarf])
            target = group_position;
        else
            target = current_goal.pos;
        std::deque<move> path = get_path(dwarf_info.pos, dwarf_info.accroche, target);
        for (move m : path) {
            if (m.release) {
                if (lacher(dwarf) != OK)
                    goto end;
            }
            if (m.hold) {
                if (agripper(dwarf) != OK)
                    goto end;
            }
            if (m.mine != ERREUR_DIRECTION) {
                for (int i = 0; i < m.strokes; i++)
                    if (miner(dwarf, m.mine) != OK)
                        goto end;
            }
            if (m.movement != ERREUR_DIRECTION) {
                if (deplacer(dwarf, m.movement) != OK)
                    goto end;
            }
        }
        end:;
    }
}

void partie_fin() {
}
